﻿using System;
using System.Threading;

namespace Curse.Friends.Statistics
{
    public class ReportingStatCounter
    {
        private readonly DateTime _startTimestamp;
        private long _totalCount;
        private long _countSinceLastSnapshot;
        private long _snapshotTimestamp;

        public ReportingStatCounter()
        {
            _startTimestamp = DateTime.UtcNow;
            _snapshotTimestamp = _startTimestamp.Ticks;
        }

        public void Track()
        {
            Interlocked.Increment(ref _totalCount);
            Interlocked.Increment(ref _countSinceLastSnapshot);
        }

        public void Track(long count)
        {
            Interlocked.Add(ref _totalCount, count);
            Interlocked.Add(ref _countSinceLastSnapshot, count);
        }

        public ReportingStatSnapshot GetSnapshot()
        {
            var total = Interlocked.Read(ref _totalCount);
            var sinceSnapshot = Interlocked.Exchange(ref _countSinceLastSnapshot, 0);

            var now = DateTime.UtcNow;
            var lastSnapshotTicks = Interlocked.Exchange(ref _snapshotTimestamp, now.Ticks);
            var lastSnapshotTime = DateTime.FromBinary(lastSnapshotTicks);

            return new ReportingStatSnapshot(total, sinceSnapshot, now, now - lastSnapshotTime, _startTimestamp);
        }

        public class ReportingStatSnapshot
        {
            public DateTime StartTimestamp { get; }

            public long TotalCount { get; }

            public long SnapshotCount { get; }

            public DateTime SnapshotTimestamp { get; }

            public TimeSpan SnapshotDuration { get; }

            internal ReportingStatSnapshot(long total, long snapshot, DateTime snapshotTimestamp, TimeSpan snapshotDuration, DateTime startTimestamp)
            {
                StartTimestamp = startTimestamp;
                TotalCount = total;
                SnapshotCount = snapshot;
                SnapshotTimestamp = snapshotTimestamp;
                SnapshotDuration = snapshotDuration;
            }
        }
    }
}
