﻿using System;
using System.Data.SqlClient;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.Extensions;
using Curse.Friends.MicroService;
using Curse.Friends.StatsWebService.Configuration;
using Curse.Friends.StatsWebService.Contracts;
using Curse.Logging;

namespace Curse.Friends.StatsWebService.Controllers
{

    [RoutePrefix("stats")]
    [AuthenticationFilter(AuthenticationLevel.ApiKey)]
    public class StatsController : MicroServiceController
    {
        
        [HttpPost]
        [Route("realtime")]
        [ResponseType(typeof(void))]
        public IHttpActionResult Realtime(StatsPayload payload)
        {
            payload.Validate();
            StatsManager.QueueRealtime(payload);              
            return Ok();
        }

        [HttpPost]
        [Route("periodic")]
        [ResponseType(typeof(void))]
        public IHttpActionResult Periodic(StatsPayload payload)
        {
            payload.Validate();
            StatsManager.QueuePeriodic(payload);                        
            return Ok();
        }

        private bool IsAuthorizedApiUser
        {
            get
            {
                return StatsWebServiceConfiguration.Current.AuthorizedUsers.Contains(Token.UserID);
            }
            
        }

        [HttpGet]
        [Route("host-stats")]
        [ResponseType(typeof(HostStats[]))]
        [AuthenticationFilter]
        public IHttpActionResult GetHostStats(long startTimestamp = 0)
        {
            if (!IsAuthorizedApiUser)
            {
                return Forbidden("You are not an authorized API user");
            }

            var start = DateTime.UtcNow.AddHours(-1);
            if (startTimestamp > 0)
            {
                try
                {
                    start = startTimestamp.FromEpochMilliconds();
                }
                catch
                {
                    return BadRequest("Invalid timestamps.");
                }

                if (start < new DateTime(2014, 1, 1))
                {
                    return BadRequest("Invalid timestamps.");
                }
            }
            return Ok(StatsManager.GetPeriodicStats(start));
        }

        [HttpGet]
        [Route("aggregate-stats")]
        [ResponseType(typeof(AggregateStats[]))]
        [AuthenticationFilter]
        public IHttpActionResult GetAggretateStats(long startTimestamp = 0)
        {
            if (!IsAuthorizedApiUser)
            {
                return Forbidden("You are not an authorized API user");
            }

            var start = DateTime.UtcNow.AddHours(-1);
            if (startTimestamp > 0)
            {
                try
                {
                    start = startTimestamp.FromEpochMilliconds();
                }
                catch
                {
                    return BadRequest("Invalid timestamps.");
                }

                if (start < new DateTime(2014, 1, 1))
                {
                    return BadRequest("Invalid timestamps.");
                }
            }
            return Ok(StatsManager.GetAggregateStats(start));
        }
        
        [HttpPost]
        [Route("regional")]
        [ResponseType(typeof(void))]
        public IHttpActionResult Regional(StatsPayload payload)
        {
            payload.Validate();
            StatsManager.QueueRegional(payload);
            return Ok();
        }

        [HttpPost]
        [Route("status")]
        [ResponseType(typeof(void))]
        public IHttpActionResult Status(HostStatusContract request)
        {
            request.Validate();

            try
            {
                // Upsert the data
                using (var conn = GetDatabaseConnection())
                {
                    using (var cmd = conn.CreateCommand())
                    {
                        cmd.CommandText = "UpdateHostStatus";
                        cmd.CommandType = System.Data.CommandType.StoredProcedure;
                        cmd.Parameters.AddWithValue("@RegionID", request.RegionID);
                        cmd.Parameters.AddWithValue("@Name", request.HostName);
                        cmd.Parameters.AddWithValue("@Type", request.HostTypeID);
                        cmd.Parameters.AddWithValue("@Status", request.HostStatus);
                        cmd.ExecuteNonQuery();
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
            
            return Ok();
        }
        
        private static SqlConnection GetDatabaseConnection()
        {
            try
            {
                var conn = new SqlConnection(StatsWebServiceConfiguration.Current.DatabaseConnectionString);
                conn.Open();
                return conn;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to open database connection: " + StatsWebServiceConfiguration.Current.DatabaseConnectionString);
                throw;
            }
        }
    }
}