﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;
using System.Linq;
using Curse.Friends.StatsWebService.Configuration;
using Curse.Friends.StatsWebService.Contracts;
using Curse.Logging;
using System.Runtime.Caching;
using Curse.Friends.Statistics.Models;
using Newtonsoft.Json;
using System.Threading;
using Curse.Extensions;

namespace Curse.Friends.StatsWebService
{
    public static class StatsManager
    {
        private static SqlConnection GetDatabaseConnection()
        {
            try
            {
                var conn = new SqlConnection(StatsWebServiceConfiguration.Current.DatabaseConnectionString);
                conn.Open();
                return conn;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to open database connection.");
                throw;
            }
        }

        private static bool _running = true;

        public static void Start()
        {
            new Thread(StatsManagerThread) { IsBackground = true }.Start();
            new Thread(RealtimeStatsManagerThread) { IsBackground = true }.Start();            
        }

        public static void Stop()
        {
            _running = false;
        }

        private static void StatsManagerThread()
        {
            while (_running)
            {               
                try
                {
                    StatsQueueItem item;
                    while (_statsQueue.TryDequeue(out item))
                    {
                        if (item.TrySave())
                        {
                            _statsQueue.Enqueue(item);
                        }
                    }
                    

                    Thread.Sleep(100);
                }
                catch (ThreadAbortException) { }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Fatal error in stats manager!");
                }

            }
        }

        private static void RealtimeStatsManagerThread()
        {
            while (_running)
            {
                try
                {
                    
                    var keys = _realtimeStats.Keys.ToArray();

                    foreach (var key in keys)
                    {
                        StatsQueueItem item;
                        if(_realtimeStats.TryRemove(key, out item))
                        {
                            item.TrySave();
                        }
                    }
                    
                    Thread.Sleep(100);
                }
                catch (ThreadAbortException) { }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Fatal error in stats manager!");
                }

            }
        }

        private abstract class StatsQueueItem
        {
            private static readonly ObjectCache DefaultCache = MemoryCache.Default;
            protected StatsPayload _payload;
            protected int _failureCount = 0;
            

            protected static int GetHostTypeID(SqlConnection conn, int hostTypeID, string hostTypeName)
            {
                var cacheKey = "GetHostTypeID-TypeName:" + hostTypeName + ";HostType: " + hostTypeID;
                var hostID = DefaultCache.Get(cacheKey);

                if (hostID == null)
                {


                    using (var cmd = conn.CreateCommand())
                    {
                        cmd.CommandText = "UpdateHostType";
                        cmd.CommandType = System.Data.CommandType.StoredProcedure;
                        cmd.Parameters.AddWithValue("@TypeID", hostTypeID);
                        cmd.Parameters.AddWithValue("@TypeName", hostTypeName);
                        hostID = cmd.ExecuteScalar();
                    }


                    DefaultCache.Add(cacheKey, hostID, new CacheItemPolicy { Priority = CacheItemPriority.NotRemovable });
                }

                return (int)hostID;
            }

            protected static int GetHostID(SqlConnection conn, string hostName, int regionID, int hostType)
            {
                var cacheKey = "GetHostID-MachineName:" + hostName + ";HostType: " + hostType + ";Region:" + regionID;
                var hostID = DefaultCache.Get(cacheKey);

                if (hostID == null)
                {
                    using (var cmd = conn.CreateCommand())
                    {
                        cmd.CommandText = "UpdateHost";
                        cmd.CommandType = System.Data.CommandType.StoredProcedure;
                        cmd.Parameters.AddWithValue("@Name", hostName);
                        cmd.Parameters.AddWithValue("@RegionID", regionID);
                        cmd.Parameters.AddWithValue("@Type", hostType);
                        hostID = cmd.ExecuteScalar();
                    }
                    DefaultCache.Add(cacheKey, hostID, new CacheItemPolicy { Priority = CacheItemPriority.NotRemovable });
                }

                return (int)hostID;
            }

            public bool TrySave()
            {
                try
                {
                    SaveToDatabase();
                }
                catch (Exception ex)
                {
                    ++_failureCount;
                    if (!WillRequeue)
                    {
                        Logger.Error(ex);
                    }
                    return WillRequeue;
                }

                return false;
            }

            protected abstract bool WillRequeue { get; }
            protected abstract void SaveToDatabase();
        }

        private class RealtimeStatsQueueItem : StatsQueueItem
        {
            public RealtimeStatsQueueItem(StatsPayload payload)
            {
                _payload = payload;
            }

            protected override bool WillRequeue
            {
                get { return _failureCount < 2; }
            }

            protected override void SaveToDatabase()
            {
                using (var conn = GetDatabaseConnection())
                {
                    // Get the host type ID
                    var hostTypeID = GetHostTypeID(conn, _payload.HostTypeID, _payload.HostTypeName);

                    // Get the host ID
                    var hostID = GetHostID(conn, _payload.HostName, _payload.RegionID, hostTypeID);

                    // Upsert the data                                
                    using (var cmd = conn.CreateCommand())
                    {
                        cmd.CommandText = "UpdateRealtimeStats";
                        cmd.CommandType = System.Data.CommandType.StoredProcedure;
                        cmd.Parameters.AddWithValue("@HostID", hostID);
                        cmd.Parameters.AddWithValue("@Stats", _payload.StatsJson);
                        cmd.ExecuteNonQuery();
                    }
                }
            }
        }

        private class PeriodicStatsQueueItem : StatsQueueItem
        {
            public PeriodicStatsQueueItem(StatsPayload payload)
            {
                _payload = payload;
            }

            protected override bool WillRequeue
            {
                get { return _failureCount < 100; }
            }

            protected override void SaveToDatabase()
            {
                using (var conn = GetDatabaseConnection())
                {
                    var hostTypeID = GetHostTypeID(conn, _payload.HostTypeID, _payload.HostTypeName);

                    // Get the host ID
                    var hostID = GetHostID(conn, _payload.HostName, _payload.RegionID, hostTypeID);

                    var stats = JsonConvert.DeserializeObject<FriendsHostStats>(_payload.StatsJson);

                    if (stats.StartDate.Second != 0 || (stats.StartDate.Minute != 0 && stats.StartDate.Minute != 15 && stats.StartDate.Minute != 30 && stats.StartDate.Minute != 45))
                    {
                        Logger.Warn("Failed to process uploaded periodic stats. The StartDate property is invalid!", stats.StartDate);                        
                    }
                    
                    // Upsert the data                   
                    using (var cmd = conn.CreateCommand())
                    {
                        cmd.CommandText = "UpdatePeriodicStats";
                        cmd.CommandType = System.Data.CommandType.StoredProcedure;
                        cmd.Parameters.AddWithValue("@HostID", hostID);
                        cmd.Parameters.AddWithValue("@StartDate", stats.StartDate);
                        cmd.Parameters.AddWithValue("@Stats", _payload.StatsJson);
                        cmd.ExecuteNonQuery();
                    }

                }
            }
        }

        private class RegionalStatsQueueItem : StatsQueueItem
        {
            public RegionalStatsQueueItem(StatsPayload payload)
            {
                _payload = payload;
            }

            protected override bool WillRequeue
            {
                get { return _failureCount < 100; }
            }

            protected override void SaveToDatabase()
            {
                var stats = JsonConvert.DeserializeObject<FriendsRegionStats>(_payload.StatsJson);
                if (stats.StartDate.Second != 0 || (stats.StartDate.Minute != 0 && stats.StartDate.Minute != 15 && stats.StartDate.Minute != 30 && stats.StartDate.Minute != 45))
                {
                    Logger.Warn("Failed to process uploaded region stats. The StartDate property is invalid!", stats.StartDate);
                    return;
                }

                // Upsert the data
                using (var conn = GetDatabaseConnection())
                {
                    using (var cmd = conn.CreateCommand())
                    {
                        cmd.CommandText = "UpdateRegionalStats";
                        cmd.CommandType = System.Data.CommandType.StoredProcedure;
                        cmd.Parameters.AddWithValue("@RegionID", _payload.RegionID);
                        cmd.Parameters.AddWithValue("@StartDate", stats.StartDate);
                        cmd.Parameters.AddWithValue("@Stats", _payload.StatsJson);
                        cmd.ExecuteNonQuery();
                    }
                }
            }
        }

        private static readonly ConcurrentDictionary<string, StatsQueueItem> _realtimeStats = new ConcurrentDictionary<string, StatsQueueItem>();
        private static readonly ConcurrentQueue<StatsQueueItem> _statsQueue = new ConcurrentQueue<StatsQueueItem>();

        private static DateTime _lastLoggedQueueFull = DateTime.UtcNow;
        private const int MaxQueueSize = 2000;

        private static void QueuePayload(StatsQueueItem queueItem)
        {
            if (_statsQueue.Count > MaxQueueSize)
            {
                if (DateTime.UtcNow.Subtract(_lastLoggedQueueFull) > TimeSpan.FromMinutes(1))
                {
                    _lastLoggedQueueFull = DateTime.UtcNow;

                    Logger.Warn("Stats queue is full with " + MaxQueueSize + " items!");
                }

                return;
            }

            _statsQueue.Enqueue(queueItem);
        }

        public static void QueueRealtime(StatsPayload payload)
        {
            if (_realtimeStats.ContainsKey(payload.HostName))
            {
                return;
            }

            _realtimeStats.TryAdd(payload.HostName, new RealtimeStatsQueueItem(payload));
        }

        public static void QueuePeriodic(StatsPayload payload)
        {
            QueuePayload(new PeriodicStatsQueueItem(payload));
        }

        public static void QueueRegional(StatsPayload payload)
        {
            QueuePayload(new RegionalStatsQueueItem(payload));
        }

        public static HostStats[] GetPeriodicStats(DateTime startDate)
        {
            var stats = new List<HostStats>();
            using (var conn = GetDatabaseConnection())
            {
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = "GetPeriodicStats";
                    cmd.CommandType = CommandType.StoredProcedure;
                    cmd.Parameters.AddWithValue("@StartDate", startDate);

                    using (var reader = cmd.ExecuteReader())
                    {
                        while (reader.Read())
                        {

                            try
                            {
                                var statsJson = (string)reader["StatsJson"];
                                var hostStats = JsonConvert.DeserializeObject<FriendsHostStats>(statsJson);
                                stats.Add(new HostStats
                                {
                                    HostID = (int)reader["HostID"],
                                    HostName = reader.SafeGetString("HostName"),
                                    RegionID = (int) reader["RegionID"],
                                    HostTypeID = (int)reader["HostTypeID"],
                                    HostTypeSlug = reader.SafeGetString("HostTypeSlug"),
                                    HostTypeName = reader.SafeGetString("HostTypeName"),
                                    HostStatus = (int)reader["HostStatus"],
                                    Stats = hostStats,
                                    Timestamp = ((DateTime)reader["StartDate"]).ToEpochMilliseconds()
                                });
                            }
                            catch (Exception ex)
                            {
                                Logger.Warn(ex, "Failed to deserialize stats from database");
                            }
                        }
                    }
                }
            }

            return stats.ToArray();
        }

        public static AggregateStats[] GetAggregateStats(DateTime startDate)
        {
            var stats = new List<AggregateStats>();
            using (var conn = GetDatabaseConnection())
            {
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = "GetRegionalStatsSinceDate";
                    cmd.CommandType = CommandType.StoredProcedure;
                    cmd.Parameters.AddWithValue("@StartDate", startDate);

                    using (var reader = cmd.ExecuteReader())
                    {
                        while (reader.Read())
                        {

                            try
                            {
                                var statsJson = (string)reader["StatsJson"];
                                var hostStats = JsonConvert.DeserializeObject<FriendsRegionStats>(statsJson);
                                stats.Add(new AggregateStats
                                {
                                  
                                    RegionID = (int)reader["RegionID"],
                                    Stats = hostStats,
                                    Timestamp = ((DateTime)reader["StartDate"]).ToEpochMilliseconds()
                                });
                            }
                            catch (Exception ex)
                            {
                                Logger.Warn(ex, "Failed to deserialize stats from database");
                            }
                        }
                    }
                }
            }

            return stats.ToArray();
        }
    }
}