﻿using System;
using System.Net;
using System.Web;
using System.Web.Mvc;
using Curse.Friends.Data;
using Curse.Friends.MicroService;
using Curse.Friends.SyncWebService.Utilities;
using Curse.Logging;
using Newtonsoft.Json;

namespace Curse.Friends.SyncWebService.Controllers
{
    public abstract class BaseCallbackController<TSeed> : Controller where TSeed : CallbackSeed
    {
        protected abstract bool CustomCallbackActions(string code, string state, User user, UserRegion userRegion, TSeed seed);

        protected ActionResult Callback(LogCategory logger, string code, string state, string error)
        {
            CallbackAccessResult<TSeed> authenticateResult = null;
            try
            {
                authenticateResult = Authenticate(logger, state, error);
                if (authenticateResult.ErrorCode >= 400)
                {
                    return RedirectToAction("OAuthCallbackFailure", new
                    {
                        errorCode = authenticateResult.ErrorCode,
                        errorMessage = authenticateResult.ErroMessage,
                        redirectUrl = authenticateResult.Seed != null ? authenticateResult.Seed.FailureRedirectUrl : null
                    });
                }

                var success = CustomCallbackActions(code, state, authenticateResult.User, authenticateResult.UserRegion, authenticateResult.Seed);

                if (!success)
                {
                    return RedirectToAction("OAuthCallbackFailure", new {redirectUrl = authenticateResult.Seed.FailureRedirectUrl, errorCode = 400});
                }


                return RedirectToAction("OAuthCallbackSuccess", new {redirectUrl = authenticateResult.Seed.SuccessRedirectUrl});
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception");
                return RedirectToAction("OAuthCallbackFailure", new
                {
                    errorCode = authenticateResult == null ? 500 : authenticateResult.ErrorCode,
                    errorMessage = authenticateResult == null ? null : authenticateResult.ErroMessage,
                    redirectUrl = authenticateResult == null ? null : authenticateResult.Seed == null ? null : authenticateResult.Seed.FailureRedirectUrl
                });
            }
        }

        protected CallbackAccessResult<TSeed> Authenticate(LogCategory logger, string state, string error)
        {
            var result = new CallbackAccessResult<TSeed> {ErrorCode = 500};

            // Get the seed
            var oauthState = OAuthState.GetLocal(state);
            if (oauthState == null || DateTime.UtcNow - oauthState.DateCreated > TimeSpan.FromMinutes(5))
            {
                logger.Info("State did not match");
                result.ErrorCode = 403;
                return result;
            }

            try
            {
                result.Seed = JsonConvert.DeserializeObject<TSeed>(oauthState.Data);
            }
            catch (JsonSerializationException)
            {
                logger.Info("Seed could not be deserialized");
                result.ErrorCode = 403;
                return result;
            }

            if (error != null)
            {
                logger.Warn("Error returned by callback", error);
                result.ErrorCode = 403;
                result.ErroMessage = error;
                return result;
            }

            if (state == null)
            {
                logger.Warn("State not returned by callback");
                result.ErrorCode = 403;
                return result;
            }

            if (result.Seed.UserID > 0)
            {
                result.UserRegion = UserRegion.GetLocal(result.Seed.UserID);
                if (result.UserRegion == null)
                {
                    logger.Warn("User's Region was not found!");
                    result.ErrorCode = 403;
                    return result;
                }

                var accessToken = TemporaryAccessToken.Get(result.UserRegion.RegionID, result.Seed.UserID);
                if (accessToken == null || accessToken.Status != TemporaryAccessTokenStatus.Active || accessToken.DateExpires < DateTime.UtcNow)
                {
                    logger.Info("Access token is not active", accessToken);
                    result.ErrorCode = 403;
                    return result;
                }
                accessToken.Status = TemporaryAccessTokenStatus.Used;
                accessToken.Update(t => t.Status);
            }
            else if (!AuthenticationContext.Current.IsAnonymous)
            {
                result.UserRegion = UserRegion.GetLocal(AuthenticationContext.Current.UserID);
                if (result.UserRegion == null)
                {
                    logger.Warn("No user region found for user " + AuthenticationContext.Current.UserID);
                    result.ErrorCode = 403;
                    result.ErroMessage = "User data error";
                    return result;
                }
            }
            else
            {
                result.ErrorCode = 403;
                return result;
            }

            result.User = Data.User.Get(result.UserRegion.RegionID, result.UserRegion.UserID);
            if (result.User == null)
            {
                logger.Warn("User could not be found", result.UserRegion);
                result.ErrorCode = 404;
                return result;
            }

            result.ErrorCode = 200;
            return result;
        }

        public ActionResult OAuthCallbackFailure(string redirectUrl, int errorCode, string errorMessage)
        {
            if (redirectUrl != null)
            {
                return Redirect(redirectUrl);
            }
            return new HttpStatusCodeResult(errorCode < 400 ? HttpStatusCode.Forbidden : (HttpStatusCode)errorCode, errorMessage);
        }

        public ActionResult OAuthCallbackSuccess(string redirectUrl)
        {
            if (redirectUrl != null)
            {
                return Redirect(redirectUrl);
            }
            return Content("Success");
        }
    }
}