﻿using System.Linq;
using System.Net;
using System.Web.Mvc;
using Curse.Friends.BattleNet;
using Curse.Friends.Data;
using Curse.Friends.SyncWebService.Configuration;
using Curse.Friends.SyncWebService.Utilities;
using Curse.Logging;

namespace Curse.Friends.SyncWebService.Controllers
{
    public class BattleNetCallbackController : BaseCallbackController<BattleNetCallbackSeed>
    {
        private static readonly LogCategory Logger = new LogCategory("BattleNetCallbackController");

        [HttpGet]
        public ActionResult Callback(string code = null, string state = null)
        {
            return base.Callback(Logger, code, state, null);
        }

        protected override bool CustomCallbackActions(string code, string state, User user, UserRegion userRegion, BattleNetCallbackSeed seed)
        {
            var tokenResult = BattleNetApiHelper.GetToken(SyncServiceConfiguration.Instance.BattleNet.LocalRegion, code, seed.ClientID,
                string.Join(" ", SyncServiceConfiguration.Instance.BattleNet.Scopes));

            if (tokenResult.StatusCode != HttpStatusCode.OK)
            {
                Logger.Warn("Unable to obtain token from BattleNet", new{tokenResult, user.UserID, user.Username});
                return false;
            }

            if (tokenResult.Content == null || tokenResult.Content.Scope == null)
            {
                Logger.Warn("Token from BattleNet has no content or no scope", new{tokenResult, user.UserID, user.Username});
                return false;
            }

            var userResult = BattleNetApiHelper.GetUser(SyncServiceConfiguration.Instance.BattleNet.LocalRegion, tokenResult.Content.AccessToken);
            if (userResult.StatusCode != HttpStatusCode.OK)
            {
                Logger.Warn("Unable to obtain user information from BattleNet", new{userResult, user.UserID, user.Username});
                return false;
            }

            var scopes = tokenResult.Content.Scope.Split(' ');
            if (scopes.All(scope => scope != "wow.profile"))
            {
                Logger.Warn("User did not give oauth permission for World of Warcraft", new { user, userRegion });
                return false;
            }

            // Create/Link account
            var account = BattleNetModelHelper.CreateAccount(tokenResult.Content, userResult.Content);
            if (account == null)
            {
                return false;
            }

            try
            {
                var updated = BattleNetModelHelper.SyncAccount(account, seed.GameRegion, user.UserID);
                account.MapUser(userRegion.UserID, userRegion.RegionID, updated);
                return true;
            }
            catch (BattleNetResponseException ex)
            {
                Logger.Warn(ex, ex.Message, new {ex.RegionInfo, ex.Content});
                return false;
            }
        }
    }
}