﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.Aerospike;
using Curse.Friends.BattleNet;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Enums;
using Curse.Friends.MicroService;
using Curse.Friends.NotificationContracts;
using Curse.Friends.SyncWebService.Configuration;
using Curse.Friends.SyncWebService.Contracts;
using Curse.Friends.SyncWebService.Utilities;
using Curse.Logging;
using Newtonsoft.Json;
using Curse.Friends.MicroService.Filters;

namespace Curse.Friends.SyncWebService.Controllers
{
    [RoutePrefix("linked-accounts")]
    public class LinkedAccountsController : MicroServiceController
    {
        /// <summary>
        /// Gets all of the authenticated user's synced accounts.
        /// </summary>
        /// <returns>The account information for the synced accounts.</returns>
        [Route("")]
        [HttpGet]
        [ResponseType(typeof(IEnumerable<ExternalAccountContract>))]
        public IHttpActionResult GetMyAccounts()
        {
            var accountMappings = ExternalAccountMapping.GetAllLocal(m => m.UserID, Token.UserID);
            var accounts = ExternalAccount.MultiGetLocal(accountMappings.Where(a => !a.IsDeleted).Select(p => new KeyInfo(p.ExternalID, p.Type)));
            var eligibleCommunities = ExternalCommunity.MultiGetLocal(accounts.Select(p => new KeyInfo(p.ExternalID, p.Type))).Where(p => p.Followers >= FriendsServiceConfiguration.Instance.VanityUrlFollowerRequirement);
            var accountContracts = accounts.Select(p =>
                p.ToContract(eligibleCommunities.Any(c => c.Type == p.Type && c.ExternalID == p.ExternalID),
                    accountMappings.FirstOrDefault(m => m.Type == p.Type && m.ExternalID == p.ExternalID)));
            return Ok(accountContracts);
        }

        [Route("token")]
        [HttpPost]
        [ResponseType(typeof(TemporaryAccessTokenContract))]
        [SocialBanFilter]
        public IHttpActionResult CreateTemporaryAccessToken()
        {
            var me = GetCurrentUserAndRegion();
            var token = TemporaryAccessToken.Get(me.Region.RegionID);
            if (token == null)
            {
                token = new TemporaryAccessToken
                {
                    UserID = me.User.UserID,
                    Token = Guid.NewGuid(),
                    DateExpires = DateTime.UtcNow.AddMinutes(10),
                    Status = TemporaryAccessTokenStatus.Active
                };
                token.Insert(me.Region.RegionID);
            }
            else
            {
                token.Token = Guid.NewGuid();
                token.DateExpires = DateTime.UtcNow.AddMinutes(10);
                token.Status = TemporaryAccessTokenStatus.Active;
                token.Update(t => t.Token, t => t.Status, t => t.DateExpires);
            }

            return Ok(token.ToNotification());
        }

        [Route("{type}")]
        [HttpGet]
        [ResponseType(typeof (string))]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        [SocialBanFilter]
        public IHttpActionResult LinkAccountWithToken(AccountType type, int userID, Guid accessToken, bool? forceVerify=null, string successRedirectUrl=null, string failureRedirectUrl=null, int? gameRegion=0)
        {
            var me = GetUserAndRegion(userID);
            var token = TemporaryAccessToken.Get(me.Region.RegionID, userID);
            if (token.Token != accessToken)
            {
                return Forbidden();
            }

            return DoLinkAccount(type, forceVerify, successRedirectUrl, failureRedirectUrl, userID, accessToken, gameRegion??0);
        }

        /// <summary>
        /// Associate an external account with the authenticated user's Curse Account.
        /// </summary>
        /// <param name="type">The external account type</param>
        /// <param name="request">Options for linking the account.</param>
        [Route("{type}")]
        [HttpPost]
        [ResponseType(typeof(void))]
        [AuthenticationFilter(AuthenticationLevel.LoggedIn, true)]
        [SocialBanFilter]
        public IHttpActionResult LinkAccount(AccountType type, [FromBody] LinkExternalAccountRequest request)
        {
            return DoLinkAccount(type, request.ForceVerify, request.SuccessRedirectUrl, request.FailureRedirectUrl, 0, Guid.Empty, request.GameRegion ?? 0);
        }

        private IHttpActionResult DoLinkAccount(AccountType type, bool? forceVerify, string successRedirectUrl, string failureRedirectUrl, int userID = 0, Guid accessToken = default(Guid),
            int gameRegion = 0)
        {
            switch (type)
            {
                case AccountType.Twitch:
                    return LinkTwitch(forceVerify, successRedirectUrl, failureRedirectUrl, userID, accessToken);
                case AccountType.WorldOfWarcraft:
                    return LinkBattleNet(successRedirectUrl, failureRedirectUrl, userID, accessToken, gameRegion);
                default:
                    return BadRequest("Unsupported external account type");
            }
        }

        /// <summary>
        /// Dissociate an external account from the authenticated user's Curse Account.
        /// </summary>
        /// <param name="type">The external account type</param>
        /// <param name="accountID">The Twitch User Name to dissociate.</param>
        [Route("links/{type}/{accountID}")]
        [HttpDelete]
        [ResponseType(typeof (void))]
        public IHttpActionResult UnlinkAccount(AccountType type, string accountID)
        {
            var me = GetCurrentUserAndRegion();

            var account = ExternalAccount.GetLocal(accountID, type);
            if (account == null)
            {
                return NotFound();
            }

            if (account.MergedUserID > 0)
            {
                Logger.Warn("User attempted to unlink a merged account", new { Token.UserID, account.ExternalID, account.ExternalUsername });
                return Forbidden("This account has been merged, and cannot be unlinked.");
            }
            

            account.DeleteMapping(me.User.UserID);

            var accountsCount = ExternalAccountMapping.GetAllLocal(a => a.UserID, me.User.UserID).Count(a => !a.IsDeleted);
            if (accountsCount == 0)
            {
                me.User.HasSyncedAccount = false;
                me.User.Update(u => u.HasSyncedAccount);
            }

            var memberships = GroupMember.GetDictionaryByUserID(me.User.UserID, false, 5000);
            var groupsToUpdate = new HashSet<Guid>();

            switch (type)
            {
                case AccountType.Twitch:
                {
                    me.User.UpdateTwitchID(null, account);
                    var mappings = ExternalCommunityMapping.GetAllLocal(c => c.ExternalID, accountID).Where(t => t.Type == type && !t.IsDeleted).ToArray();
                    foreach (var mappingToUpdate in mappings.Where(m => memberships.ContainsKey(m.GroupID)))
                    {
                        groupsToUpdate.Add(mappingToUpdate.GroupID);
                    }
                    break;
                }
                case AccountType.WorldOfWarcraft:
                {
                    var guildInfos = new HashSet<ExternalGuildIdentifier>(ExternalGuildMember.GetAllLocal(m => m.AccountID, accountID).GroupBy(m => m.GetGuildInfo()).Select(m => m.Key));
                    var guilds = ExternalGuild.MultiGetLocal(guildInfos.Select(g => new KeyInfo(g.Type, g.GameRegion, g.GameServer, g.Name)));
                    var mappings = ExternalGuildMapping.MultiGetLocal(guilds.SelectMany(g => g.MappedGroups.Select(id => new KeyInfo(g.Type, g.GameRegion, g.GameServer, g.Name, id))));
                    foreach (var mappingToUpdate in mappings.Where(m => memberships.ContainsKey(m.GroupID)))
                    {
                        groupsToUpdate.Add(mappingToUpdate.GroupID);
                    }
                    break;
                }
                default:
                    return BadRequest("Unsupported account type");
            }

            var groups = Group.MultiGetLocal(groupsToUpdate.Select(id => new KeyInfo(id))).ToDictionary(g => g.GroupID);
            
            foreach (var groupToUpdate in groupsToUpdate)
            {
                Group group;
                if (groups.TryGetValue(groupToUpdate, out group))
                {
                    GroupMemberWorker.CreateAccountUnlinked(group, me.User.UserID);
                }
            }

            return StatusCode(HttpStatusCode.NoContent);
        }

        [Route("resync")]
        [HttpPost]
        [ResponseType(typeof(AccountResyncResponse))]
        [SocialBanFilter]
        public IHttpActionResult Resync()
        {
            try
            {
                var me = GetCurrentUserAndRegion();

                var throttle = TimeSpan.FromSeconds(Data.User.ManualAccountSyncThrottleSeconds);
                var timeSinceSync = DateTime.UtcNow - me.User.LastManualAccountSync;
                if (timeSinceSync < throttle)
                {
                    return BadRequest(new AccountResyncResponse
                    {
                        Status = AccountResyncStatus.Throttled,
                        RetryAfterMilliseconds = (long) (throttle - timeSinceSync).TotalMilliseconds
                    });
                }

                me.User.LastManualAccountSync = DateTime.UtcNow;
                me.User.Update(u => u.LastManualAccountSync);

                // Intentionally including deleted linked accounts to properly remove roles as well
                var linkedAccounts = ExternalAccountMapping.GetAllLocal(a => a.UserID, me.User.UserID);
                foreach (var account in linkedAccounts)
                {
                    switch (account.Type)
                    {
                        case AccountType.Twitch:
                            ExternalUserSyncWorker.Create(me.User.UserID, account.ExternalID, account.Type);
                            break;
                        case AccountType.WorldOfWarcraft:
                            ExternalGuildUserSyncWorker.Create(me.User.UserID, account.Type);
                            break;
                    }
                }

                return Ok(new AccountResyncResponse
                {
                    Status = AccountResyncStatus.Success,
                    RetryAfterMilliseconds = (long) throttle.TotalMilliseconds
                });
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to resync external accounts for " + Token.UserID);
                return BadRequest(new AccountResyncResponse
                {
                    Status = AccountResyncStatus.Error,
                    RetryAfterMilliseconds = 0L
                });
            }
        }

        #region Twitch

        private IHttpActionResult LinkTwitch(bool? forceVerify, string successRedirectUrl, string failureRedirectUrl, int userID, Guid accessToken)
        {
            var seed = new TwitchCallbackSeed
            {
                Seed = DateTime.UtcNow.GetHashCode(),
                SuccessRedirectUrl = successRedirectUrl,
                FailureRedirectUrl = failureRedirectUrl,
                UserID = userID,
                AccessToken = accessToken
            };

            var state = OAuthState.CreateForSeed(seed);
            var authRedirect = string.Format("https://api.twitch.tv/kraken/oauth2/authorize?response_type=code&client_id={0}&redirect_uri={1}&scope={2}&force_verify={3}&state={4}",
                FriendsServiceConfiguration.Instance.TwitchClientID,
                SyncServiceConfiguration.Instance.Twitch.RedirectUri,
                string.Join("+", SyncServiceConfiguration.Instance.Twitch.Scopes),
                (forceVerify ?? false).ToString().ToLower(),
                state.Key);

            return Redirect(TwitchApi.TwitchApiHelper.Default.GetAuthRedirect(forceVerify??false, SyncServiceConfiguration.Instance.Twitch.RedirectUri, SyncServiceConfiguration.Instance.Twitch.Scopes, state.Key));
        }

        #endregion

        #region BattleNet

        private IHttpActionResult LinkBattleNet(string successRedirectUrl, string failureRedirectUrl, int userID, Guid accessToken, int gameRegion)
        {
            if (!Enum.IsDefined(typeof (BattleNetRegion), gameRegion))
            {
                return BadRequest("Unsupported Game Region");
            }
            var app = BattleNetApiHelper.GetApp();
            var seed = new BattleNetCallbackSeed
            {
                Seed = DateTime.UtcNow.GetHashCode(),
                SuccessRedirectUrl = successRedirectUrl,
                FailureRedirectUrl = failureRedirectUrl,
                UserID = userID,
                AccessToken = accessToken,
                ClientID = app.ClientID,
                GameRegion = gameRegion
            };
            var state = OAuthState.CreateForSeed(seed);

            var regionInfo = BattleNetApiHelper.GetRegionInfo((BattleNetRegion) gameRegion);

            var authRedirect = string.Format("{0}/oauth/authorize?response_type=code&client_id={1}&redirect_uri={2}&scope={3}&state={4}",
                regionInfo.OAuthBaseUrl,
                app.ClientID,
                app.RedirectUri,
                string.Join("+", SyncServiceConfiguration.Instance.BattleNet.Scopes),
                state.Key);
            return Redirect(authRedirect);
        }

        #endregion
    }
}