﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.Aerospike;
using Curse.Friends.Data;
using Curse.Friends.Data.Search;
using Curse.Friends.Enums;
using Curse.Friends.MicroService;
using Curse.Friends.NotificationContracts;
using Curse.Friends.SyncWebService.Contracts;
using Curse.Friends.TwitchApi;
using Curse.Logging;
using Curse.Extensions;
using Curse.Friends.MicroService.Filters;
using Curse.Friends.Configuration;

namespace Curse.Friends.SyncWebService.Controllers
{
    [RoutePrefix("linked-communities")]
    public class LinkedCommunitiesController : MicroServiceController
    {

        /// <summary>
        /// Gets the emotes associated with the synced community.
        /// </summary>T
        /// <param name="communityID">The ID of the community</param>
        /// <returns></returns>
        [HttpPost]
        [Route("{communityID}/resync")]
        [ResponseType(typeof(void))]
        [SocialBanFilter]
        public IHttpActionResult Resync(string communityID)
        {
            // Ensure the user has access
            var group = Group.GetLocal(communityID);
            if (group == null || group.Status == GroupStatus.Deleted)
            {
                return NotFound();
            }

            if (!group.IsRootGroup)
            {
                return BadRequest();
            }

            var throttle = TimeSpan.FromSeconds(Group.ManualSyncThrottleSeconds);
            if (DateTime.UtcNow - group.DateLastManualSync < throttle)
            {
                return TooManyRequests();
            }

            // Get all external links for this community
            var externalAccounts = group.GetExternalAccounts();
            if (!externalAccounts.Any())
            {
                return BadRequest("No accounts attached to this community.");
            }
            var communities = ExternalCommunity.MultiGetLocal(externalAccounts.Select(a => new KeyInfo(a.ExternalID, a.Type)));

            foreach (var account in externalAccounts)
            {
                switch (account.Type)
                {
                    case AccountType.Twitch:
                        TwitchAccountSyncWorker.Create(account.ExternalID, true);
                        break;
                }

                var community = communities.FirstOrDefault(c => c.ExternalID == account.ExternalID && c.Type == account.Type);
                if (community == null || community.MappedGroups.Count == 0)
                {
                    continue;
                }
                ExternalCommunityMemberSyncWorker.Create(community.RegionID, community.ExternalID, community.Type);
            }

            return Ok();
        }

        /// <summary>
        /// Gets the emotes associated with the synced community.
        /// </summary>T
        /// <param name="type">The type of community</param>
        /// <param name="communityID">The ID of the community</param>
        /// <returns></returns>
        [HttpGet]
        [Route("{type}/{communityID}/emotes")]
        [ResponseType(typeof(IEnumerable<ExternalCommunityEmoticonContract>))]
        public IHttpActionResult GetSyncedEmotes(AccountType type, string communityID)
        {
            return Ok(ExternalCommunityEmoticon.GetAllByTypeAndSyncID(type, communityID).Select(e => e.ToNotification()));
        }

        /// <summary>
        /// Dissociates a community from a Curse group.
        /// </summary>
        /// <param name="type">The type of community</param>
        /// <param name="groupID">The ID of the Curse group</param>
        /// <param name="communityID">The ID of the community</param>
        [HttpPost]
        [Route("{type}/{communityID}/links/{groupID}")]
        [ResponseType(typeof(void))]
        [SocialBanFilter]
        public IHttpActionResult Link(AccountType type, Guid groupID, string communityID)
        {
            var userID = Token.UserID;

            var community = ExternalCommunity.GetLocal(communityID, type);
            if (community == null)
            {
                return NotFound();
            }

            var group = Group.GetWritable(groupID);
            if (group == null || group.Status == GroupStatus.Deleted)
            {
                return NotFound();
            }

            group.LinkExternalCommunity(userID, community);

            return Ok(new LinkExternalCommunityResponse
            {
                StreamID = communityID,
                Type = type,
                Followers = community.Followers,
                Subscribers = community.Subscribers
            });
        }

        /// <summary>
        /// Dissociates a community from a Curse group.
        /// </summary>
        /// <param name="type">The type of community</param>
        /// <param name="groupID">The ID of the Curse group</param>
        /// <param name="communityID">The ID of the community</param>
        [HttpDelete]
        [Route("{type}/{communityID}/links/{groupID}")]
        [ResponseType(typeof(void))]
        public IHttpActionResult Unlink(AccountType type, Guid groupID, string communityID)
        {
            var group = Group.GetWritable(groupID);
            if (group == null || group.Status == GroupStatus.Deleted)
            {
                return NotFound();
            }

            group.UnlinkExternalCommunity(Token.UserID, communityID, type);

            return StatusCode(HttpStatusCode.NoContent);
        }

        [HttpGet]
        [Route("{type}/{communityID}/links")]
        [ResponseType(typeof(IEnumerable<GroupNotification>))]
        public IHttpActionResult GetLinkedServers(AccountType type, string communityID)
        {
            var community = ExternalCommunity.GetLocal(communityID, type);
            if(community == null || community.MappedGroups == null || community.MappedGroups.Count == 0)
            {
                return Ok(new GroupNotification[0]);
            }

            var groups = Group.MultiGetLocal(community.MappedGroups.Select(id => new KeyInfo(id)));
            var memberships = GroupMember.MultiGetLocal(community.MappedGroups.Select(id => new KeyInfo(id, Token.UserID))).Where(m=>!m.IsDeleted).ToDictionary(m => m.GroupID);
            return Ok(groups.Select(g =>
            {
                var notification = g.ToNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl, false, true, true);
                notification.Membership = memberships.GetValueOrDefault(g.GroupID)?.ToMembershipNotification(g);
                return notification;
            }));
        }

        [HttpGet]
        [Route("{groupID}")]
        [ResponseType(typeof(IEnumerable<ExternalCommunityLinkContract>))]
        public IHttpActionResult GetLinkedCommunities(Guid groupID)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            var member = group.CheckPermission(GroupPermissions.AccessAdminPanel, Token.UserID);

            var mappings = ExternalCommunityMapping.GetAllLocal(m => m.GroupID, groupID).Where(c => !c.IsDeleted).ToArray();
            var communities = ExternalCommunity.MultiGetLocal(mappings.Select(m => new KeyInfo(m.ExternalID, m.Type)));
            var accounts = ExternalAccountMapping.GetAllLocal(a => a.UserID, member.UserID).Where(a => !a.IsDeleted).ToArray();

            var contracts = new List<ExternalCommunityLinkContract>();
            foreach (var mapping in mappings)
            {
                var community = communities.FirstOrDefault(c => c.ExternalID == mapping.ExternalID && c.Type == mapping.Type);
                if (community == null)
                {
                    Logger.Debug("Community not found for a mapped community!");
                    continue;
                }
                //accounts.Any(a=>a.ExternalID==community.ExternalID && a.Type == community.Type)
                contracts.Add(mapping.ToContract(community));
            }
            return Ok(contracts);
        }

        [HttpPost]
        [Route("{groupID}/{type}/{communityID}/settings")]
        [ResponseType(typeof(ExternalCommunityLinkContract))]
        [SocialBanFilter]
        public IHttpActionResult ChangeCommunitySettings(Guid groupID, AccountType type, string communityID, [FromBody] ChangeCommunitySettingsRequest request)
        {
            var community = ExternalCommunity.GetLocal(communityID, type);
            if (community == null)
            {
                return NotFound();
            }

            var mapping = ExternalCommunityMapping.GetLocal(groupID, communityID, type);
            if (mapping == null || mapping.IsDeleted)
            {
                return NotFound();
            }

            mapping.ChangeSettings(request.SyncEmoticons, request.SyncEvents, request.GracePeriodAction, request.GracePeriodDays);

            return Ok(mapping.ToContract(community));
        }

        [Route("memberships")]
        [HttpGet]
        [ResponseType(typeof(SyncedCommunityMembershipResponse))]
        public IHttpActionResult GetCommunityMemberships()
        {
            var userID = Token.UserID;

            // get all accounts for all types
            var myAccounts = ExternalAccountMapping.GetAllLocal(m => m.UserID, userID).Where(m => !m.IsDeleted);

            // get all relevant memberships
            var allMyMemberships = myAccounts.SelectMany(a => ExternalCommunityMembership.GetAllLocal(m => m.UserAndTypeIndex, a.ExternalID + a.Type))
                .Where(m => m.Status != ExternalCommunityMembershipStatus.Deleted && (m.RoleTag == GroupRoleTag.SyncedSubscriber || m.RoleTag == GroupRoleTag.SyncedFollower))
                .ToLookup(g => new Tuple<string, AccountType>(g.ExternalCommunityID, g.Type));

            var communities = ExternalCommunity.MultiGetLocal(allMyMemberships.Select(m => new KeyInfo(m.Key.Item1, m.Key.Item2)));
            var mappings = ExternalCommunityMapping.MultiGetLocal(communities.SelectMany(c => c.MappedGroups.Select(g => new KeyInfo(g, c.ExternalID, c.Type))));
            var allGroups = Group.MultiGetLocal(mappings.GroupBy(m => m.GroupID).Select(m => new KeyInfo(m.Key)));
            var allMemberships = GroupMember.MultiGetLocal(allGroups.Select(g => new KeyInfo(g.GroupID, userID))).ToDictionary(m => m.GroupID);
            var suggestions = Enum.GetValues(typeof(AccountType)).Cast<AccountType>().ToDictionary(v => v, v => new List<SyncedServerMembership>());

            var mappingsLookup = mappings.ToLookup(g => g.GroupID);
            foreach (var group in allGroups)
            {
                foreach (var mapping in mappingsLookup[group.GroupID])
                {
                    var bestMembership = allMyMemberships[new Tuple<string, AccountType>(mapping.ExternalID, mapping.Type)].OrderByDescending(m => m.RoleTag).First();
                    suggestions[mapping.Type].Add(new SyncedServerMembership
                    {
                        GroupID = group.GroupID,
                        GroupMemberCount = group.MemberCount,
                        GroupTitle = group.Title,
                        IsPublic = group.IsPublic,

                        ExternalCommunityID = mapping.ExternalID,
                        ExternalCommunityType = mapping.Type,
                        ExternalCommunityName = mapping.ExternalDisplayName ?? mapping.ExternalName,

                        RoleTag = bestMembership.RoleTag,
                        IsMember = allMemberships.ContainsKey(group.GroupID)
                    });
                }
            }

            return Ok(new SyncedCommunityMembershipResponse
            {
                TwitchSuggestions = suggestions[AccountType.Twitch].ToArray(),
                YouTubeSuggestions = suggestions[AccountType.YouTube].ToArray()
            });
        }

        [HttpGet]
        [Route("{type}/{communityID}/subscribers")]
        [ResponseType(typeof(PremiumMemberSearchResponse))]
        public IHttpActionResult GetPremiumMembers(AccountType type, string communityID, string query = null, Guid? groupID = null, int? pageSize = null, int? pageNumber = null)
        {
            if (pageSize.HasValue)
            {
                if (pageSize.Value < 1)
                {
                    return BadRequest("PageSize must be at least 1");
                }

                if (pageSize.Value > 25)
                {
                    return BadRequest("PageSize cannot exceed 25");
                }
            }

            if (pageNumber.HasValue && pageNumber.Value < 0)
            {
                return BadRequest("PageNumber must be non-negative");
            }


            var community = ExternalCommunity.GetLocal(communityID, type);
            if (community == null)
            {
                return NotFound();
            }

            Group group = null;


            if (groupID.HasValue)
            {
                group = Group.GetByID(groupID.Value);

                if (group == null || !group.HasPermission(GroupPermissions.AccessAdminPanel, Token.UserID))
                {
                    return Forbidden();
                }

                // Get the synced Twitch owner role for this account
                var link = ExternalAccountMapping.GetLocal(Token.UserID, type, communityID);
                if (link == null || link.IsDeleted)
                {
                    // Only allow users linked to the account view subscribers.
                    return Forbidden();
                }
            }
            else
            {
                var link = ExternalAccountMapping.GetLocal(Token.UserID, type, communityID);
                if (link == null || link.IsDeleted)
                {
                    // Only allow users linked to the account view subscribers.
                    return Forbidden();
                }
            }


            long actives;
            long inactives;
            ExternalCommunityMemberManager.CountByStatus(communityID, type, GroupRoleTag.SyncedSubscriber, out actives, out inactives);

            var results = ExternalCommunityMemberManager.Search(new ExternalCommunityMemberSearch
            {
                PageSize = pageSize,
                PageNumber = pageNumber,
                ExternalCommunityID = communityID,
                ExternalCommunityType = type,
                ExternalCommunityRoles = new[] { GroupRoleTag.SyncedSubscriber },
                Query = query,
                MinEntryNumber = 1
            });


            IReadOnlyCollection<GroupMember> groupMembers = new GroupMember[0];
            var userStatistics = new Dictionary<int, UserStatistics>();

            if (group != null)
            {
                var userIDs = new HashSet<int>(results.SelectMany(r => r.UserIDs).ToArray());
                groupMembers = GroupMember.MultiGetLocal(results.SelectMany(r => r.UserIDs.Select(id => new KeyInfo(groupID.Value, id))));
                userStatistics = UserStatistics.GetDictionaryByUserIDs(userIDs);
            }

            var groupMemberLookup = groupMembers.ToLookup(g => g.UserID);
            var contracts = new List<ExternalCommunityMemberContract>();
            foreach (var result in results)
            {
                GroupMember member = null;
                foreach (var userID in result.UserIDs)
                {
                    member = groupMemberLookup[userID].FirstOrDefault();
                    if (member != null)
                    {
                        break;
                    }
                }

                contracts.Add(result.ToNotification(member, member != null ? userStatistics.GetValueOrDefault(member.UserID) : null));
            }
            return Ok(new PremiumMemberSearchResponse
            {
                Active = actives,
                Inactive = inactives,
                Results = contracts.ToArray()
            });
        }

        #region Twitch

        [HttpPost]
        [Route("twitch/{communityID}/follow")]
        [ResponseType(typeof(FollowTwitchChannelResponse))]
        [SocialBanFilter]
        public IHttpActionResult FollowChannel(string communityID)
        {
            var community = ExternalCommunity.GetLocal(communityID, AccountType.Twitch);
            if (community == null)
            {
                return NotFound();
            }

            var externalMappings = ExternalAccountMapping.GetAllLocal(m => m.UserID, Token.UserID).Where(m => m.Type == AccountType.Twitch && !m.IsDeleted);

            var successes = new List<string>();
            var failures = new List<string>();
            foreach (var mapping in externalMappings)
            {
                var account = ExternalAccount.GetLocal(mapping.ExternalID, mapping.Type);
                if (account == null)
                {
                    failures.Add(mapping.ExternalID);
                    continue;
                }

                try
                {
                    var followResponse = TwitchApiHelper.GetClient(account.ClientID).FollowChannel(mapping.ExternalID, communityID, account.AuthToken);
                    if (followResponse.Status == TwitchResponseStatus.Success)
                    {
                        community.AddMemberRole(account.ExternalID, GroupRoleTag.SyncedFollower, account.ExternalUsername, account.ExternalDisplayName, followResponse.Value.CreatedAt);
                        successes.Add(mapping.ExternalID);
                    }
                    else
                    {
                        failures.Add(mapping.ExternalID);
                    }
                }
                catch (Exception ex)
                {
                    Logger.Warn(ex, "Error while trying to follow a twitch channel", new { Token.UserID, communityID });
                    failures.Add(mapping.ExternalID);
                }
            }


            return Ok(new FollowTwitchChannelResponse
            {
                SuccessfulAccounts = successes.ToArray(),
                UnsuccessfulAccounts = failures.ToArray()
            });
        }

        #endregion
    }
}