﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Data;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Enums;
using Curse.Friends.MicroService;
using Curse.Friends.SyncWebService.Contracts;
using Curse.Friends.MicroService.Filters;

namespace Curse.Friends.SyncWebService.Controllers
{
    [RoutePrefix("linked-guilds")]
    public class LinkedGuildsController : MicroServiceController
    {
        [HttpPost]
        [Route("{type}/{region}/{server}/{guildName}/links/{groupID}")]
        [SocialBanFilter]
        public IHttpActionResult LinkGuild(AccountType type, int region, string server, string guildName, Guid groupID, LinkGuildRequest request)
        {
            switch (type)
            {
                case AccountType.WorldOfWarcraft:
                    return LinkWowGuild(groupID, region, server, guildName, request);
                default:
                    return BadRequest("Unsupported guild type");
            }
        }

        [HttpDelete]
        [Route("{type}/{region}/{server}/{guildName}/links/{groupID}")]
        public IHttpActionResult UnlinkGuild(AccountType type, int region, string server, string guildName, Guid groupID)
        {
            switch (type)
            {
                case AccountType.WorldOfWarcraft:
                    return UnlinkWowGuild(groupID, region, server, guildName);
                default:
                    return BadRequest("Unsupported guild type");
            }
        }

        [HttpGet]
        [Route("{type}")]
        [ResponseType(typeof(IEnumerable<GuildDetailsContract>))]
        public IHttpActionResult GetMyGuilds(AccountType type)
        {
            var accounts = ExternalAccountMapping.GetAllLocal(m => m.UserID, Token.UserID).Where(m => !m.IsDeleted && m.Type == type);

            var memberships = accounts.SelectMany(a => ExternalGuildMember.GetAllLocal(m => m.AccountID, a.ExternalID))
                .GroupBy(m => m.GetGuildInfo())
                .Where(g=>g.Key!=null)
                .ToDictionary(g => g.Key, g => g.ToLookup(l => l.GuildRole));

            var guildInfos = new HashSet<ExternalGuildIdentifier>(memberships.Keys);
            var guilds = ExternalGuild.MultiGetLocal(guildInfos.Select(g => new KeyInfo(g.Type, g.GameRegion, g.GameServer, g.Name)));

            var allDetails = guilds.Select(g => new GuildDetailsContract
            {
                GuildInfo = g.ToContract()
            });

            return Ok(allDetails);
        }

        #region Battle.Net



        private IHttpActionResult LinkWowGuild(Guid groupID, int gameRegion, string gameServer, string name, LinkGuildRequest request)
        {
            var guild = ExternalGuild.Get(new ExternalGuildIdentifier(AccountType.WorldOfWarcraft, gameRegion, gameServer, name));
            if (guild == null)
            {
                return NotFound();
            }

            var group = Group.GetWritableByID(groupID);
            if (group == null)
            {
                return BadRequest();
            }

            var guildRoles = new List<NewGuildSyncRole>();
            var roles = request.Roles.ToDictionary(r => r.Tag);
            foreach (var rank in ExternalGuildRole.GetRoleTags(AccountType.WorldOfWarcraft))
            {
                var roleInfo = roles.GetValueOrDefault(rank);
                guildRoles.Add(new NewGuildSyncRole
                {
                    Tag = rank,
                    Name = roleInfo != null ? roleInfo.Name : null,
                    HasBadge = false,
                    IsOfficer = false,
                    VanityColor = 0,
                    VanityBadge = 0
                });
            }
            group.LinkExternalGuild(Token.UserID, guild, guildRoles.ToArray());
            return StatusCode(HttpStatusCode.Accepted);
        }

        private IHttpActionResult UnlinkWowGuild(Guid groupID, int gameRegion, string gameServer, string name)
        {
            var guild = ExternalGuild.Get(new ExternalGuildIdentifier(AccountType.WorldOfWarcraft, gameRegion, gameServer, name));
            if (guild == null)
            {
                return NotFound();
            }

            var group = Group.GetWritableByID(groupID);
            if (group == null)
            {
                return BadRequest();
            }

            group.UnlinkExternalGuild(Token.UserID, guild);

            return StatusCode(HttpStatusCode.NoContent);
        }

        #endregion
    }
}