﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Curse.Extensions;
using Curse.Friends.Data;
using Curse.Logging;

namespace Curse.Friends.Tracing
{
    public class FilteredUserLogger
    {
        private readonly LogCategory TracedUserLogger;
        private readonly LogCategory ThrottledLogger;



        public FilteredUserLogger(string logCategory)
        {
            TracedUserLogger = new LogCategory(logCategory) { ReleaseLevel = LogLevel.Debug, AlphaLevel = LogLevel.Trace};
            ThrottledLogger = new LogCategory(logCategory) { ReleaseLevel = LogLevel.Debug, AlphaLevel = LogLevel.Trace, Throttle = TimeSpan.FromMinutes(1) };
        }

        public LogLevel AlphaLevel
        {
            get { return TracedUserLogger.AlphaLevel; }
            set
            {
                TracedUserLogger.AlphaLevel = value;
                ThrottledLogger.AlphaLevel = value;
            }
        }

        public LogLevel ReleaseLevel
        {
            get { return TracedUserLogger.ReleaseLevel; }
            set
            {
                TracedUserLogger.ReleaseLevel = value;
                ThrottledLogger.ReleaseLevel = value;
            }
        }

        public void Info(string message, object data = null)
        {
            TracedUserLogger.Info(message, data);
        }

        public void Error(Exception ex, string message, object data = null)
        {
            TracedUserLogger.Error(ex, message, data);
        }

        public void Error(User user, Exception ex, string message, object data = null)
        {
            if (TracedUsers.IsDeveloper(user))
            {
                message = message + " (" + user.Username + ")";                
                TracedUserLogger.Error(ex, message, data);
                return;
            }

            TracedUserLogger.Error(ex, message, data);
        }

        public void Log(User user, string message, object data = null, Func<object> traceUserData = null)
        {
            Log(user, null, message, data, traceUserData);
        }

        public void Log(IEnumerable<string> twitchUserIDs, string message, object data = null, Func<object> traceUserData = null)
        {
            if (TracedUsers.IsWhitelisted(twitchUserIDs))
            {
                var logData = traceUserData == null ? data : new { data, userData = traceUserData() };
                TracedUserLogger.Trace(message, logData);
                return;
            }

            // Otherwise, throttled logging for a sample
            ThrottledLogger.Trace(message, data);
        }

        public void Log(ExternalCommunity community, string message, object data = null, Func<object> traceUserData = null)
        {
            // If the user is 'developer', go with verbose Debug logging
            if (TracedUsers.IsDeveloper(community?.ExternalID))
            {
                message = $"{message} ({community?.ExternalName})";
                var logData = traceUserData == null ? data : new { data, userData = traceUserData() };
                TracedUserLogger.Debug(message, logData);
                return;
            }

            // If the user is 'interesting', go with verbose logging
            if (TracedUsers.IsInterestingUser(null, null, community))
            {
                var logData = traceUserData == null ? data : new { data, userData = traceUserData() };
                TracedUserLogger.Trace(message, logData);
                return;
            }

            // Otherwise, throttled logging for a sample
            ThrottledLogger.Trace(message, data);
        }

        public void Log(User user, ExternalAccount externalAccount, string message, object data = null, Func<object> traceUserData = null)
        {
            // If the user is 'developer', go with verbose Debug logging
            if (TracedUsers.IsDeveloper(user))
            {
                message = message + " (" + user.Username + ")";
                var logData = traceUserData == null ? data : new { data, userData = traceUserData() };
                TracedUserLogger.Debug(message, logData);
                return;
            }

            // If the user is 'interesting', go with verbose logging
            if (TracedUsers.IsInterestingUser(user, externalAccount, null))
            {
                var logData = traceUserData == null ? data : new { data, userData = traceUserData() };
                TracedUserLogger.Trace(message, logData);
                return;
            }

            // Otherwise, throttled logging for a sample
            ThrottledLogger.Trace(message, data);
        }

        public void Log(string twitchUserID, string message, object data = null)
        {
            // If the user is 'developer', go with verbose Debug logging
            if (TracedUsers.IsDeveloper(twitchUserID))
            {
                message = $"{message} (TID: {twitchUserID})";
                TracedUserLogger.Debug(message, data);
                return;
            }
            
            // If the user is 'interesting', go with verbose logging
            if (TracedUsers.IsWhitelisted(twitchUserID))
            {
                TracedUserLogger.Trace(message, data);
                return;
            }

            // Otherwise, throttled logging for a sample
            ThrottledLogger.Trace(message, data);
        }

        public void Log(int curseUserID, string message, object data = null)
        {
            if (TracedUsers.IsDeveloper(curseUserID))
            {
                TracedUserLogger.Debug(message, data);
                return;
            }

            // If the user is 'interesting', go with verbose logging
            if (TracedUsers.IsWhitelisted(curseUserID))
            {
                TracedUserLogger.Trace(message, data);
                return;
            }

            // Otherwise, throttled logging for a sample
            ThrottledLogger.Trace(message, data);
        }

        public void Warn(User user, string message, object data = null)
        {
            Warn(user, null, message, data);
        }



        public void Warn(User user, ExternalAccount externalAccount, string message, object data = null)
        {
            // If the user is 'interesting', go with verbose logging
            if (TracedUsers.IsInterestingUser(user, externalAccount, null))
            {
                TracedUserLogger.Warn(message, data);
                return;
            }

            // Otherwise, throttled logging for a sample
            ThrottledLogger.Warn(message, data);
        }

        public void Warn(Exception ex, User user, ExternalAccount externalAccount, string message, object data = null)
        {
            // If the user is 'interesting', go with verbose logging
            if (TracedUsers.IsInterestingUser(user, externalAccount, null))
            {
                TracedUserLogger.Warn(ex, message, data);
                return;
            }

            // Otherwise, throttled logging for a sample
            ThrottledLogger.Warn(ex, message, data);
        }

        public void Warn(string twitchUserID, string message, object data = null)
        {
            // If the user is 'interesting', go with verbose logging
            if (TracedUsers.IsWhitelisted(twitchUserID))
            {
                TracedUserLogger.Warn(message, data);
                return;
            }

            // Otherwise, throttled logging for a sample
            ThrottledLogger.Warn(message, data);
        }

        public void Warn(int curseUserID, string message, object data = null)
        {
            // If the user is 'interesting', go with verbose logging
            if (TracedUsers.IsWhitelisted(curseUserID))
            {
                TracedUserLogger.Warn(message, data);
                return;
            }

            // Otherwise, throttled logging for a sample
            ThrottledLogger.Warn(message, data);
        }
    }
}
