﻿using System;
using System.Collections.Generic;
using System.Net.Http;
using System.Text;
using Newtonsoft.Json;
using System.Net.Http.Headers;
using System.Threading.Tasks;
using Curse.Friends.TwitchApi.CurseShim;
using System.Threading;

namespace Curse.Friends.TwitchApi
{
    public class CurseShimClient : IDisposable
    {
        private static readonly TimeSpan _spamClassifierTimeout = TimeSpan.FromMilliseconds(500);

        private readonly HttpClient _client;
        private readonly string _baseUrl;

        private static readonly JsonSerializerSettings SerializerSettings = new JsonSerializerSettings
        {
            NullValueHandling = NullValueHandling.Ignore
        };

        public CurseShimClient(string authToken, string baseUrl, int timeoutMilliseconds = 30000)
        {
            _client = new HttpClient { Timeout = TimeSpan.FromMilliseconds(timeoutMilliseconds) };

            if (!baseUrl.EndsWith("/"))
            {
                baseUrl = baseUrl + "/";
            }

            _baseUrl = baseUrl;
            if (authToken != null)
            {
                _client.DefaultRequestHeaders.Authorization = new AuthenticationHeaderValue("Bearer", authToken);
            }

        }

        public TwitchResponse<IsFamiliarResponse> IsFamiliar(string userID, string otherUserID)
        {
            return Get<IsFamiliarResponse>("users/{0}/is_familiar/{1}", userID, otherUserID);
        }

        public TwitchResponse<CheckAccessResponse> CheckAccess()
        {
            return Get<CheckAccessResponse>("check_access");
        }

        public TwitchResponse<VerifiedBotsResponse> VerifiedBots()
        {
            return Get<VerifiedBotsResponse>("verified_bots");
        }

        public TwitchResponse<MergeAccountPayload> MergeAccount(MergeAccountPayload request)
        {
            return Post<MergeAccountPayload>("merge_account", request);
        }

        public TwitchResponse<ErrorableResponse> CreateReport(string fromUserId, string targetUserId, string content, string reason, string description)
        {
            var report = new CreateReportRequest
            {
                FromUserID = fromUserId,
                TargetUserID = targetUserId,
                Content = content,
                Reason = reason,
                Description = description
            };

            return Post<ErrorableResponse>("create_report", report);
        }

        public TwitchResponse<ClassifySpamResponse> ClassifySpam(string senderId, string sendeLogin, string senderIp, string recipientId, string recipientLogin, string body)
        {
            var request = new ClassifySpamRequest
            {
                SenderID = senderId,
                SenderLogin = sendeLogin,
                SenderIp = senderIp,
                RecipientID = recipientId,
                RecipientLogin = recipientLogin,
                Body = body
            };

            return Post<ClassifySpamResponse>("classify_spam", request, _spamClassifierTimeout);
        }

        public TwitchResponse<ChannelModsResponse> GetChannelMods(string channelID)
        {
            return Get<ChannelModsResponse>("channels/{0}/mods", channelID);
        }

        public TwitchResponse<ErrorableResponse> SendChattersList(string serverID, Dictionary<string,string[]> chatters)
        {
            var request = new ChattersListRequest
            {
                ServerID = serverID,
                Chatters = chatters
            };
            return Patch<ErrorableResponse>("chatter_list", request);
        }

        private TwitchResponse<T> ParseResponse<T>(string url, HttpResponseMessage response) where T : ErrorableResponse
        {
            var statusCode = (int)response.StatusCode;
            T content;
            string responseContent = null;
            try
            {
                responseContent = response.Content.ReadAsStringAsync().Result;
                content = JsonConvert.DeserializeObject<T>(responseContent, SerializerSettings);
            }
            catch (JsonException)
            {
                return new TwitchResponse<T>(url, statusCode >= 500 ? TwitchResponseStatus.TwitchServerError : TwitchResponseStatus.InvalidResponseFormat, statusCode, responseContent);
            }

            if (content == null)
            {
                // Bypass invalid format for a 204 which may not have content
                if(statusCode == 204)
                {
                    return new TwitchResponse<T>(url, null, TwitchResponseStatus.Success);
                }

                return new TwitchResponse<T>(url, TwitchResponseStatus.InvalidResponseFormat, statusCode);
            }

            if (!response.IsSuccessStatusCode)
            {
                switch (statusCode)
                {
                    case 401:
                        return new TwitchResponse<T>(url, content, TwitchResponseStatus.Unauthorized, statusCode);
                    case 404:
                        return new TwitchResponse<T>(url, content, TwitchResponseStatus.NotFound, statusCode);
                    case 422:
                        return new TwitchResponse<T>(url, content, TwitchResponseStatus.Unprocessable, statusCode);
                }

                if (statusCode >= 500)
                {
                    return new TwitchResponse<T>(url, content, TwitchResponseStatus.TwitchServerError, statusCode);
                }

                return new TwitchResponse<T>(url, content, TwitchResponseStatus.OtherNonSuccess, statusCode);
            }

            if (string.IsNullOrEmpty(content.Error))
            {
                return new TwitchResponse<T>(url, content, TwitchResponseStatus.Success);
            }

            if (string.IsNullOrEmpty(content.Status))
            {
                return new TwitchResponse<T>(url, content, TwitchResponseStatus.InvalidResponseFormat);
            }

            int errorStatusCode;
            if (int.TryParse(content.Status, out errorStatusCode))
            {
                switch (errorStatusCode)
                {
                    case 401:
                        return new TwitchResponse<T>(url, content, TwitchResponseStatus.Unauthorized, errorStatusCode);
                    case 404:
                        return new TwitchResponse<T>(url, content, TwitchResponseStatus.NotFound, errorStatusCode);
                    case 422:
                        return new TwitchResponse<T>(url, content, TwitchResponseStatus.Unprocessable, errorStatusCode);
                }

                if (errorStatusCode >= 500)
                {
                    return new TwitchResponse<T>(url, content, TwitchResponseStatus.TwitchServerError, errorStatusCode);
                }
            }

            return new TwitchResponse<T>(url, content, TwitchResponseStatus.OtherNonSuccess);
        }

        private TwitchResponse<T> Get<T>(string action) where T : ErrorableResponse
        {
            var url = _baseUrl + action;
            HttpResponseMessage response = null;
            try
            {
                response = _client.GetAsync(url).Result;
                return ParseResponse<T>(url, response);
            }
            catch (Exception ex)
            {
                return CreateResponseFromException<T>(url, response, ex);
            }
        }

        private TwitchResponse<T> Get<T>(string actionFormat, params object[] args) where T : ErrorableResponse
        {
            var url = _baseUrl + string.Format(actionFormat, args);
            HttpResponseMessage response = null;
            try
            {
                response = _client.GetAsync(url).Result;
                return ParseResponse<T>(url, response);
            }
            catch (Exception ex)
            {
                return CreateResponseFromException<T>(url, response, ex);
            }
        }

        private static TwitchResponse<T> CreateResponseFromException<T>(string url, HttpResponseMessage response, Exception ex)
        {
            var exception = ex;
            var agg = ex as AggregateException;
            if (agg != null)
            {
                if (ex.InnerException != null)
                {
                    exception = ex.InnerException;
                }
                else if (agg.InnerExceptions != null && agg.InnerExceptions.Count > 0)
                {
                    exception = agg.InnerExceptions[0];
                }
            }
            var statusCode = response == null ? 0 : (int)response.StatusCode;
            
            var status = TwitchResponseStatus.GeneralError;

            if (statusCode >= 500)
            {
                status = TwitchResponseStatus.TwitchServerError;
            }
            else if (exception is TaskCanceledException)
            {
                status = TwitchResponseStatus.Timeout;
            }

            return new TwitchResponse<T>(url, status, statusCode, new { exception.Message, exception.StackTrace });
        }

        private TwitchResponse<T> Post<T>(string url, Dictionary<string, string> content) where T : ErrorableResponse
        {
            HttpResponseMessage response = null;
            try
            {
                var requestContent = new FormUrlEncodedContent(content);
                response = _client.PostAsync(url, requestContent).Result;
                return ParseResponse<T>(url, response);
            }
            catch (Exception ex)
            {
                return CreateResponseFromException<T>(url, response, ex);
            }
        }

        private TwitchResponse<T> Post<T>(string action, object content, TimeSpan? timeoutOverride = null) where T : ErrorableResponse
        {
            HttpResponseMessage response = null;
            var url = _baseUrl + action;
            try
            {
                var jsonBody = JsonConvert.SerializeObject(content);
                var requestContent = new StringContent(jsonBody, Encoding.UTF8, "application/json");
                if (timeoutOverride.HasValue)
                {
                    using (var cts = new CancellationTokenSource(timeoutOverride.Value))
                    {
                        response = _client.PostAsync(url, requestContent, cts.Token).Result;
                    }
                }
                else
                {
                    response = _client.PostAsync(url, requestContent).Result;
                }
                return ParseResponse<T>(url, response);
            }
            catch (Exception ex)
            {
                return CreateResponseFromException<T>(url, response, ex);
            }
        }
        private TwitchResponse<T> Put<T>(HttpClient client, string url, object content) where T : ErrorableResponse
        {
            HttpResponseMessage response = null;
            try
            {
                var requestContent = new StringContent(JsonConvert.SerializeObject(content ?? new object()), Encoding.UTF8, "application/json");
                response = _client.PutAsync(url, requestContent).Result;
                return ParseResponse<T>(url, response);
            }
            catch (Exception ex)
            {
                return CreateResponseFromException<T>(url, response, ex);
            }
        }

        private TwitchResponse<T> Patch<T>(string action, object content) where T : ErrorableResponse
        {
            var url = $"{_baseUrl}{action}";
            HttpResponseMessage response = null;
            try
            {
                var requestContent = new StringContent(JsonConvert.SerializeObject(content ?? new object()), Encoding.UTF8, "application/json");
                var request = new HttpRequestMessage(new HttpMethod("PATCH"), url)
                {
                    Content = requestContent
                };
                response = _client.SendAsync(request).Result;
                return ParseResponse<T>(url, response);
            }
            catch (Exception ex)
            {
                return CreateResponseFromException<T>(url, response, ex);
            }
        }

        public void Dispose()
        {
            var client = _client;
            if (client != null)
            {
                client.Dispose();
            }
        }
    }
}
