﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Text;
using Newtonsoft.Json;

namespace Curse.Friends.TwitchApi
{
    public class TwitchApiHelper
    {        
        public const int MaxLimit = 100;
        
        private static readonly JsonSerializerSettings _settings = new JsonSerializerSettings
        {
            NullValueHandling = NullValueHandling.Ignore
        };


        public static void Initialize(string clientID, string clientSecret, Dictionary<string, string> additionalClients = null)
        {

            _default = new TwitchApiHelper(clientID, clientSecret);
            HelpersByClientID[clientID] = _default;

            if (additionalClients != null)
            {
                foreach (var kvp in additionalClients)
                {
                    HelpersByClientID[kvp.Key] = new TwitchApiHelper(kvp.Key, kvp.Value);
                }                
            }
        }

        public static string[] ClientIDs 
        {
            get { return HelpersByClientID.Keys.ToArray(); }
            
        }

        public static TwitchApiHelper GetClient(string id)
        {
            if (id == null)
            {
                return _default;
            }

            TwitchApiHelper client;
            if (!HelpersByClientID.TryGetValue(id, out client))
            {
                throw new InvalidOperationException("TwitchApiHelper: Attempt to get a client with an unknown id: " + id);
            }


            return client;
        }

        public static bool IsValidClientID(string id)
        {
            return HelpersByClientID.ContainsKey(id);
        }

        private static readonly Dictionary<string, TwitchApiHelper> HelpersByClientID = new Dictionary<string, TwitchApiHelper>();

        private static TwitchApiHelper _default;
        public static TwitchApiHelper Default
        {
            get
            {
                if (_default == null)
                {
                    throw new InvalidOperationException("TwitchApiHelper: Attempt to use the default Twitch API Client before it has been initialized");
                }

                return _default;
            }
        }

        private static void VerifyID(string idString)
        {
            int id;

            if (!int.TryParse(idString, out id))
            {
                throw new ArgumentException("The supplied ID argument is not a valid number: " + idString);
            }
        }

        private static void VerifyIDs(string[] idStrings)
        {
            if (!idStrings.Any())
            {
                return;
            }

            // Assume if the first element is numeric, the rest are
            VerifyID(idStrings[0]);
        }


        private readonly string _clientID;
        private readonly string _clientSecret;
        private readonly HttpClient _client;

        public TwitchApiHelper(string clientID, string clientSecret)
        {
            _clientID = clientID;
            _clientSecret = clientSecret;
            _client = new HttpClient();
            _client.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue("application/vnd.twitchtv.v5+json"));
            _client.DefaultRequestHeaders.Add("Client-ID", _clientID);
        }

        #region Syncing

        public string GetAuthRedirect(bool forceVerify, string redirectUri, string[] scopes, string state)
        {
            var authRedirect = string.Format("https://api.twitch.tv/kraken/oauth2/authorize?response_type=code&client_id={0}&redirect_uri={1}&scope={2}&force_verify={3}&state={4}",
                _clientID, redirectUri, string.Join("+", scopes), forceVerify.ToString().ToLower(), state);
            return authRedirect;
        }

        public TwitchResponse<TwitchAccessTokenResponse> GetTwitchToken(string code, string state, string redirectUri)
        {
            var body = new Dictionary<string, string>
            {
                {"client_id", _clientID},
                {"client_secret", _clientSecret},
                {"grant_type", "authorization_code"},
                {"redirect_uri", redirectUri},
                {"code", code},
            };
            if (state != null)
            {
                body["state"] = state;
            }

            using (var client = CreateClient())
            {
                return Post<TwitchAccessTokenResponse>(client, "https://api.twitch.tv/kraken/oauth2/token", body);
            }
        }

        #endregion
        
        #region Authenticated Requests

        public TwitchResponse<SubscriptionsResponse> GetSubscriptions(string channelID, string ownerToken, int offset = 0, int limit = MaxLimit, string direction = null)
        {
            VerifyID(channelID);
            return Get<SubscriptionsResponse>($"https://api.twitch.tv/kraken/channels/{channelID}/subscriptions?limit={limit}&offset={offset}&direction={direction ?? "asc"}", ownerToken);
        }

        public TwitchResponse<IReadOnlyCollection<Subscription>> GetAllSubscriptions(string channelID, string ownerToken)
        {
            VerifyID(channelID);
            var subscriptions = new List<Subscription>();

            var offset = 0;
            var limit = 100;
            int? total = null;
            do
            {
                var response = GetSubscriptions(channelID, ownerToken, offset, limit);
                if (response.Status != TwitchResponseStatus.Success)
                {
                    return new TwitchResponse<IReadOnlyCollection<Subscription>>(channelID, response.Status);
                }
                total = total ?? response.Value.Total;
                subscriptions.AddRange(response.Value.Subscriptions);
                offset += limit;
            } while (offset < total);

            return new TwitchResponse<IReadOnlyCollection<Subscription>>(channelID, subscriptions, TwitchResponseStatus.Success);
        }

        public TwitchResponse<TwitchBaseInfo> GetTwitchRootInfo(string accessToken)
        {
            return Get<TwitchBaseInfo>("https://api.twitch.tv/kraken", accessToken);
        }

        public bool TestUserSubs(string userID, string channelID, string ownerToken)
        {
            VerifyID(userID);
            VerifyID(channelID);

            return Get<Subscription>($"https://api.twitch.tv/kraken/channels/{channelID}/subscriptions/{userID}", ownerToken).Status == TwitchResponseStatus.Success;
        }

        public TwitchResponse<User> GetUser(string authToken)
        {
            return Get<User>("https://api.twitch.tv/kraken/user", authToken);
        }

        public TwitchResponse<Subscription> GetUserSubscription(string username, string channelID, string channelOwnerToken)
        {
            VerifyID(channelID);
            return Get<Subscription>($"https://api.twitch.tv/kraken/channels/{channelID}/subscriptions/{username}", channelOwnerToken);
        }

        public TwitchResponse<Follow> FollowChannel(string username, string channelID, string userAuthToken)
        {
            VerifyID(channelID);
            using (var client = CreateClient(userAuthToken))
            {
                return Put<Follow>(client, string.Format("https://api.twitch.tv/kraken/users/{0}/follows/channels/{1}", username, channelID), string.Empty);
            }
        }

        public TwitchResponse<UserEmoteResponse> GetUserEmotes(string userID, string token)
        {
            VerifyID(userID);
            return Get<UserEmoteResponse>($"https://api.twitch.tv/kraken/users/{userID}/emotes", token);
        }

        #endregion

        #region Non-Authenticated Calls

        public TwitchResponse<IReadOnlyCollection<Follow>> GetAllUserFollows(string userID)
        {
            VerifyID(userID);
            var follows = new List<Follow>();

            var offset = 0;
            var limit = 100;
            int? total = null;
            do
            {
                var response = DoGetUserFollows(userID, offset, limit);
                if (response.Status != TwitchResponseStatus.Success)
                {
                    return new TwitchResponse<IReadOnlyCollection<Follow>>(userID, response.Status, response.StatusCode, response.AdditionalInfo);
                }
                total = total ?? response.Value.Total;
                follows.AddRange(response.Value.Follows);
                offset += limit;
            } while (offset < total);

            return new TwitchResponse<IReadOnlyCollection<Follow>>(userID, follows, TwitchResponseStatus.Success);
        }

        public TwitchResponse<UserFollowsResponse> GetUserFollows(string userID, int offset = 0, int limit = MaxLimit, string direction = null)
        {
            return DoGetUserFollows(userID, offset, limit, direction);
        }

        private  TwitchResponse<UserFollowsResponse> DoGetUserFollows(string userID, int offset = 0, int limit = MaxLimit, string direction = null)
        {
            VerifyID(userID);
            return Get<UserFollowsResponse>($"https://api.twitch.tv/kraken/users/{userID}/follows/channels?limit={limit}&offset={offset}&direction={direction ?? "desc"}");
        }

        public TwitchResponse<EmoticonResponse> GetEmoticons(string channelID)
        {
            VerifyID(channelID);
            return Get<EmoticonResponse>(string.Format("https://api.twitch.tv/kraken/chat/{0}/emoticons", channelID));
        }

        public TwitchResponse<User> GetPublicUser(string username)
        {
            return Get<User>(string.Format("https://api.twitch.tv/kraken/users/{0}", username));
        }

        public TwitchResponse<Follow> GetUserChannelFollow(string userID, string channelID)
        {
            VerifyID(userID);
            return Get<Follow>($"https://api.twitch.tv/kraken/users/{userID}/follows/channels/{channelID}");
        }

        public TwitchResponse<Follow> GetUserChannelFollowV3(string username, string channelName)
        {
            using (var client = CreateClient())
            {
                client.DefaultRequestHeaders.Accept.Clear();
                client.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue("application/vnd.twitchtv.v3+json"));
                return Get<Follow>(client, $"https://api.twitch.tv/kraken/users/{username}/follows/channels/{channelName}");
            }
        }

        public bool TestUserFollows(string userID, string channelID)
        {
            return GetUserChannelFollow(userID, channelID).Status == TwitchResponseStatus.Success;
        }

        public TwitchResponse<Channel> GetChannel(string channelID)
        {
            VerifyID(channelID);
            return Get<Channel>(string.Format("https://api.twitch.tv/kraken/channels/{0}", channelID));
        }

        public TwitchResponse<Channel> GetChannelV3(string channelName)
        {
            using (var client = CreateClient())
            {
                client.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue("application/vnd.twitchtv.v3+json"));
                return Get<Channel>(client, string.Format("https://api.twitch.tv/kraken/channels/{0}", channelName));
            }
        }

        public TwitchResponse<Channel> GetChannelByID(string channelID)
        {
            VerifyID(channelID);
            return Get<Channel>(string.Format("https://api.twitch.tv/kraken/channels/{0}", channelID));
        }


        public TwitchResponse<ChannelStreamResponse> GetChannelStream(string channelID)
        {
            VerifyID(channelID);
            return Get<ChannelStreamResponse>(string.Format("https://api.twitch.tv/kraken/streams/{0}", channelID));
        }

        public TwitchResponse<StreamsResponse> GetStreams(params string[] channelIDs)
        {
            VerifyIDs(channelIDs);
            return Get<StreamsResponse>("https://api.twitch.tv/kraken/streams/?channel=" + string.Join(",", channelIDs));
        }

        public TwitchResponse<ChannelFollowsResponse> GetChannelFollowers(string channelID, int offset = 0, int limit = MaxLimit)
        {
            VerifyID(channelID);
            return Get<ChannelFollowsResponse>(string.Format("https://api.twitch.tv/kraken/channels/{0}/follows?limit={1}&offset={2}", channelID, limit, offset));
        }

        public TwitchResponse<Follower[]> GetChannelFollowersSince(string channelID, DateTime timestamp)
        {
            VerifyID(channelID);
            var followers = new Dictionary<string, Follower>();
            var status = TwitchResponseStatus.Success;
            int statusCode = 0;

            var next = string.Format("https://api.twitch.tv/kraken/channels/{0}/follows?limit={1}", channelID, MaxLimit);
            while (true)
            {
                var response = Get<ChannelFollowsResponse>(next);
                if (response.Status != TwitchResponseStatus.Success)
                {
                    status = response.Status;
                    statusCode = response.StatusCode;
                    break;
                }

                foreach (var follower in response.Value.Follows)
                {
                    if (!followers.ContainsKey(follower.User.ID))
                    {
                        followers[follower.User.ID] = follower;
                    }
                }

                if (response.Value.Follows.Length == 0)
                {
                    // No results
                    break;
                }

                if (response.Value.Follows.Where(f => f != null).Min(f => f.CreatedDate) < timestamp)
                {
                    // We've reached our stopping point
                    break;
                }

                if (next == response.Value.Links.Next)
                {
                    // Nothing more to scan
                    break;
                }

                next = response.Value.Links.Next;
                if (string.IsNullOrEmpty(next))
                {
                    status = TwitchResponseStatus.InvalidResponseFormat;
                    break;
                }
            }

            return new TwitchResponse<Follower[]>(channelID, followers.Values.ToArray(), status, statusCode);
        }

        public TwitchResponse<AllEmoticonsResponse> GetAllEmoticons()
        {
            return Get<AllEmoticonsResponse>("https://api.twitch.tv/kraken/chat/emoticons");
        }

        public TwitchResponse<EmoticonImageResponse> GetAllEmoticonImages()
        {
            return Get<EmoticonImageResponse>("https://api.twitch.tv/kraken/chat/emoticon_images");
        }

        public TwitchResponse<ChatBadgeResponse> GetChatBadges(string channelID)
        {
            VerifyID(channelID);
            return Get<ChatBadgeResponse>(string.Format("https://api.twitch.tv/kraken/chat/{0}/badges", channelID));
        }

        public TwitchResponse<GlobalChatBadgesResponse> GetGlobalChatBadges()
        {
            return Get<GlobalChatBadgesResponse>("https://badges.twitch.tv/v1/badges/global/display?language=en");
        }

        public TwitchResponse<GlobalChatBadgesResponse> GetChannelChatBadges(string externalID)
        {
            VerifyID(externalID);
            return Get<GlobalChatBadgesResponse>(string.Format("https://badges.twitch.tv/v1/badges/channels/{0}/display?language=en", externalID));
        }

        #endregion

        #region Fuel

        public TwitchResponse<FuelAuthorizationResponse> AuthorizeFuel(string authToken, string[] scopes)
        {
            using (var client = CreateClient(authToken))
            {

                var contents = new
                {
                    client_secret = _clientSecret,
                    scope = scopes
                };
                return PostJson<FuelAuthorizationResponse>(client, $"https://api.twitch.tv/v5/fuel/authorize?client_id={_clientID}", contents);
            }
        }

        #endregion

        #region Support Methods

        public HttpClient CreateClient(string authToken = null)
        {
            if (_clientID == null)
            {
                throw new InvalidOperationException("Twitch API was not initialized");
            }
            var client = new HttpClient();
            client.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue("application/vnd.twitchtv.v5+json"));
            client.DefaultRequestHeaders.Add("Client-ID", _clientID);

            if (authToken != null)
            {
                client.DefaultRequestHeaders.Authorization = new AuthenticationHeaderValue("OAuth", authToken);
            }

            return client;
        }

        private TwitchResponse<T> Get<T>(string url, string authToken = null) where T : ErrorableResponse
        {
            HttpResponseMessage response = null;
            try
            {
                var request = new HttpRequestMessage(HttpMethod.Get, url);
                if (authToken != null)
                {
                    request.Headers.Authorization = new AuthenticationHeaderValue("OAuth", authToken);
                }

                response = _client.SendAsync(request).Result;
                return ParseResponse<T>(url, response);
            }
            catch (Exception ex)
            {
                return CreateResponseFromException<T>(url, response, ex);
            }
        }

        private TwitchResponse<T> Get<T>(HttpClient client, string url) where T : ErrorableResponse
        {
            HttpResponseMessage response = null;
            try
            {
                response = client.GetAsync(url).Result;
                return ParseResponse<T>(url, response);
            }
            catch (Exception ex)
            {
                return CreateResponseFromException<T>(url, response, ex);
            }
        }

        private TwitchResponse<T> Post<T>(HttpClient client, string url, IEnumerable<KeyValuePair<string, string>> content) where T: ErrorableResponse
        {
            HttpResponseMessage response = null;
            try
            {
                var requestContent = new FormUrlEncodedContent(content);
                response = client.PostAsync(url, requestContent).Result;
                return ParseResponse<T>(url, response);
            }
            catch (Exception ex)
            {
                return CreateResponseFromException<T>(url, response, ex);
            }
        }

        private TwitchResponse<T> PostJson<T>(HttpClient client, string url, object content) where T : ErrorableResponse
        {
            HttpResponseMessage response = null;
            try
            {
                var requestContent = new StringContent(JsonConvert.SerializeObject(content), Encoding.UTF8, "application/json");
                response = client.PostAsync(url, requestContent).Result;
                return ParseResponse<T>(url, response);
            }
            catch (Exception ex)
            {
                return CreateResponseFromException<T>(url, response, ex);
            }
        }

        private TwitchResponse<T> Put<T>(HttpClient client, string url, object content) where T : ErrorableResponse
        {
            HttpResponseMessage response = null;
            try
            {
                var requestContent = new StringContent(JsonConvert.SerializeObject(content ?? new object()), Encoding.UTF8, "application/json");
                response = client.PutAsync(url, requestContent).Result;
                return ParseResponse<T>(url, response);
            }
            catch (Exception ex)
            {
                return CreateResponseFromException<T>(url, response, ex);
            }
        }

        private TwitchResponse<T> CreateResponseFromException<T>(string url, HttpResponseMessage response, Exception ex)
        {
            var exception = ex;
            var agg = ex as AggregateException;
            if (agg != null)
            {
                if (ex.InnerException != null)
                {
                    exception = ex.InnerException;
                }
                else if (agg.InnerExceptions != null && agg.InnerExceptions.Count > 0)
                {
                    exception = agg.InnerExceptions[0];
                }
            }
            var statusCode = response == null ? 0 : (int) response.StatusCode;
            return new TwitchResponse<T>(url, statusCode >= 500 ? TwitchResponseStatus.TwitchServerError : TwitchResponseStatus.GeneralError, statusCode, new {exception.Message, exception.StackTrace});
        }

        private TwitchResponse<T> ParseResponse<T>(string url, HttpResponseMessage response) where T : ErrorableResponse
        {
            var statusCode = (int)response.StatusCode;
            T content;
            string responseContent = null;
            try
            {
                responseContent = response.Content.ReadAsStringAsync().Result;
                content = JsonConvert.DeserializeObject<T>(responseContent, _settings);
            }
            catch (JsonException)
            {
                return new TwitchResponse<T>(url, statusCode >= 500 ? TwitchResponseStatus.TwitchServerError : TwitchResponseStatus.InvalidResponseFormat, statusCode, responseContent);
            }

            if (content == null)
            {
                return new TwitchResponse<T>(url, TwitchResponseStatus.InvalidResponseFormat, statusCode);
            }

            if (!response.IsSuccessStatusCode)
            {
                switch (statusCode)
                {
                    case 401:
                        return new TwitchResponse<T>(url, content, TwitchResponseStatus.Unauthorized, statusCode);
                    case 404:
                        return new TwitchResponse<T>(url, content, TwitchResponseStatus.NotFound, statusCode);
                    case 422:
                        return new TwitchResponse<T>(url, content, TwitchResponseStatus.Unprocessable, statusCode);
                }

                if (statusCode >= 500)
                {
                    return new TwitchResponse<T>(url, content, TwitchResponseStatus.TwitchServerError, statusCode);
                }

                return new TwitchResponse<T>(url, content, TwitchResponseStatus.OtherNonSuccess, statusCode);
            }

            if (string.IsNullOrEmpty(content.Error))
            {
                return new TwitchResponse<T>(url, content, TwitchResponseStatus.Success);
            }

            if (string.IsNullOrEmpty(content.Status))
            {
                return new TwitchResponse<T>(url, content, TwitchResponseStatus.InvalidResponseFormat);
            }

            int errorStatusCode;
            if (int.TryParse(content.Status, out errorStatusCode))
            {
                switch (errorStatusCode)
                {
                    case 401:
                        return new TwitchResponse<T>(url, content, TwitchResponseStatus.Unauthorized, errorStatusCode);
                    case 404:
                        return new TwitchResponse<T>(url, content, TwitchResponseStatus.NotFound, errorStatusCode);
                    case 422:
                        return new TwitchResponse<T>(url, content, TwitchResponseStatus.Unprocessable, errorStatusCode);
                }

                if (errorStatusCode >= 500)
                {
                    return new TwitchResponse<T>(url, content, TwitchResponseStatus.TwitchServerError, errorStatusCode);
                }
            }

            return new TwitchResponse<T>(url, content, TwitchResponseStatus.OtherNonSuccess);
        }

        #endregion

        #region Avatar

        public void UploadAvatar(string userID, string authToken, System.IO.Stream imageStream)
        {
            // Get upload URL
            using (var client = CreateClient(authToken))
            {
                var urlResponse = client.PostAsync($"https://api.twitch.tv/kraken/users/{userID}/upload_image?image_type=profile_image",
                    new FormUrlEncodedContent(new KeyValuePair<string, string>[0])).Result;

                if (!urlResponse.IsSuccessStatusCode)
                {
                    throw new Exception("upload_image was not successful: " + urlResponse.StatusCode);
                }

                var uploadUrl = JsonConvert.DeserializeObject<UploadImageResponse>(urlResponse.Content.ReadAsStringAsync().Result);
                // Upload stream
                var uploadResponse = _client.PutAsync(uploadUrl.UploadUrl, new StreamContent(imageStream)).Result;
                if (!uploadResponse.IsSuccessStatusCode)
                {
                    throw new Exception("Upload response was not successful: " + uploadResponse.StatusCode);
                }
            }
        }

        #endregion
    }
}
