﻿using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using System.Security.Cryptography;
using System.Text;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.AuthenticationClient;
using Curse.Friends.AuthenticationClient.AuthService;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Logging;
using User = Curse.Friends.Data.User;
using Curse.Friends.Data.Models;
using Curse.Friends.TwitchApi;

namespace Curse.Friends.TwitchIdentityMerge
{    
    public class IdentityMergeAuthHelper
    {
        private static readonly LogCategory DiagLogger = new LogCategory("IdentityMergeAuthHelper") { ReleaseLevel = LogLevel.Debug };
        private static readonly LogCategory ThrottledLogger = new LogCategory("IdentityMergeAuthHelper") { ReleaseLevel = LogLevel.Debug, Throttle = TimeSpan.FromMinutes(15) };

        private static bool HasChanges(TwitchUserInfo twitchUser, User user)
        {
            if (twitchUser.Email != user.EmailAddress)
            {
                return true;
            }

            if (twitchUser.EmailVerified != user.EmailVerified)
            {
                return true;
            }

            if (twitchUser.Username != user.Username)
            {
                return true;
            }

            if (twitchUser.DisplayName != user.DisplayName)
            {
                return true;
            }

            return false;
        }

        public static IdentityMergeState GetIdentityMergeState(TwitchUserInfo twitchUser)
        {

            MappedUserResult authResp = null;
            ExternalAccount account = null;

            try
            {
                // Check auth
                authResp = AuthenticationProvider.GetMappedUser(int.Parse(twitchUser.UserID));
                if (authResp == null || (authResp.Status == MappedUserStatus.Mapped && authResp.MappedUser == null))
                {
                    Logger.Warn("No AuthResponse in GetIdentityMergeState", new { TwitchID = twitchUser.UserID, authResp = authResp });
                    return new IdentityMergeState(IdentityMergeFailureReason.GeneralError);
                }

                if (authResp.Status == MappedUserStatus.FailedGeneralError)
                {
                    Logger.Warn("Auth response unsuccessful", new { authResp, TwitchID = twitchUser.UserID }); ;
                    return new IdentityMergeState(IdentityMergeFailureReason.GeneralError);
                }

                account = ExternalAccount.GetByTwitchUserID(twitchUser.UserID);

                // Merge/Update user if mapped
                if (authResp.Status == MappedUserStatus.Mapped)
                {
                    var userID = authResp.MappedUser.UserID;
                    if (account == null)
                    {
                        DiagLogger.Debug("ExternalAccount record does not exist, creating and merging user", new { authResp.MappedUser });
                        MergeLocalAccount(twitchUser, authResp.MappedUser.UserID, authResp.MappedUser.Mode == EUserMapMode.AutoProvisioned);
                    }
                    else if (account.IsMerged)
                    {
                        if (account.MergedUserID != authResp.MappedUser.UserID)
                        {
                            Logger.Warn("Local Account did not match Auth, unmerging local account", new { Account = account.GetLogData(), authResp.MappedUser });
                            UnmergeLocalUser(account, account.MergedUserID);
                        }

                        // Ensure that the mapped user can be retrieved
                        var userRegion = UserRegion.GetByUserID(userID);
                        var user = userRegion?.GetUser();

                        // If the region or user records are not found, re-merge
                        if (userRegion == null || user == null || user.TwitchID != twitchUser.UserID || !user.IsMerged || (authResp.MappedUser.Mode != EUserMapMode.AutoProvisioned && user.IsProvisional))
                        {
                            DiagLogger.Debug("Merging User with missing records or incorrect Twitch ID", new { authResp.MappedUser, User = user?.GetLogData(), UserRegion = userRegion });
                            MergeLocalAccount(twitchUser, userID, authResp.MappedUser.Mode == EUserMapMode.AutoProvisioned);
                            user = User.FindByUserID(userID);
                        }
                        else if (HasChanges(twitchUser, user))
                        {
                            DiagLogger.Debug("Updating local user with new name/email", new { authResp.MappedUser, User = user.GetLogData(), UserRegion = userRegion });
                            UpdateLocalAccount(twitchUser, user, userRegion);
                        }
                        else
                        {
                            DiagLogger.Debug("Updating local account with new auth info", new { authResp.MappedUser, User = user?.GetLogData(), UserRegion = userRegion });
                            UpdateExternalAccount(twitchUser, user, userRegion.RegionID, account);
                        }
                    }

                    var mergeStatus = authResp.MappedUser.Mode == EUserMapMode.AutoProvisioned ? IdentityMergeStatus.AutoProvisioned : IdentityMergeStatus.Merged;
                    return new IdentityMergeState(twitchUser.UserID, userID, twitchUser.Username, twitchUser.DisplayName, mergeStatus);
                }

                // Unmerge if merged in AS but unmerged in auth
                if (authResp.Status == MappedUserStatus.Unmapped && account != null && account.IsMerged)
                {
                    Logger.Warn("Auth thinks user is not mapped, services think it is, unmerging", new { Account = account.GetLogData() });
                    UnmergeLocalUser(account, account.MergedUserID);
                }

                // Prompt for merge
                return new IdentityMergeState(IdentityMergeStatus.Unmerged);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "failure in GetIdentityMergeState",
                    new
                    {
                        TwitchID = twitchUser.UserID,
                        TwitchName = twitchUser.Username,
                        twitchExternalAccount = account?.GetLogData(),
                        authResp = authResp
                    });
                return new IdentityMergeState(IdentityMergeFailureReason.GeneralError);
            }
        }

        private static IdentityMergeFailureReason GetFailureReason(ProvisionMappedUserStatus status)
        {
            switch (status)
            {                
                case ProvisionMappedUserStatus.FailedGeneralError:
                    return IdentityMergeFailureReason.GeneralError;
                case ProvisionMappedUserStatus.FailedTwitchAccountMerged:
                    return IdentityMergeFailureReason.TwitchAccountMerged;
                default:
                    throw new ArgumentException("Unknown status: " + status);

            }
        }

        private static IdentityMergeFailureReason GetFailureReason(MergeUserAccountStatus status)
        {
            switch (status)
            {
                case MergeUserAccountStatus.FailedCurseAccountMerged:
                    return IdentityMergeFailureReason.CurseAccountMerged;
                case MergeUserAccountStatus.FailedTwitchAccountMerged:
                    return IdentityMergeFailureReason.TwitchAccountMerged;
                case MergeUserAccountStatus.FailedDeletedAccount:
                    return IdentityMergeFailureReason.DeletedAccount;
                case MergeUserAccountStatus.FailedGeneralError:
                    return IdentityMergeFailureReason.GeneralError;
                case MergeUserAccountStatus.FailedUnknownUser:
                    return IdentityMergeFailureReason.UnknownUser;
                case MergeUserAccountStatus.FailedInvalidPassword:
                    return IdentityMergeFailureReason.InvalidPassword;
                case MergeUserAccountStatus.FailedValidationError:
                    return IdentityMergeFailureReason.ValidationError;
                default:
                    throw new ArgumentException("Unknown status: " + status);

            }
        }

        public static IdentityMergeState ProvisionOrGetMappedUser(string twitchUserID, bool isAutoProvisioned)
        {            
            var result = TwitchApiHelper.Default.GetPublicUser(twitchUserID);
            if (result.Status != TwitchResponseStatus.Success)
            {
                return new IdentityMergeState(IdentityMergeFailureReason.ApiError);
            }

            var twitchUser = new TwitchUserInfo
            {
                UserID = result.Value.ID,
                Username = result.Value.Name,
                DisplayName = result.Value.DisplayName,
                Avatar = result.Value.Logo,
                Bio = result.Value.Bio,
            };

            var mergeState = ProvisionMappedUser(twitchUser, isAutoProvisioned);
            if (mergeState.Status != IdentityMergeStatus.Failed)
            {
                return mergeState;
            }

            return GetIdentityMergeState(twitchUser);
        }

        public static IdentityMergeState ProvisionMappedUser(string twitchUserID, bool isAutoProvisioned)
        {
            var result = TwitchApiHelper.Default.GetPublicUser(twitchUserID);
            if (result.Status != TwitchResponseStatus.Success)
            {
                return new IdentityMergeState(IdentityMergeFailureReason.ApiError);
            }

            var twitchUser = new TwitchUserInfo
            {
                UserID = result.Value.ID,
                Username = result.Value.Name,
                DisplayName = result.Value.DisplayName,
                Avatar = result.Value.Logo,
                Bio = result.Value.Bio,
            };

            return ProvisionMappedUser(twitchUser, isAutoProvisioned);
        }

        public static IdentityMergeState ProvisionMappedUser(TwitchUserInfo twitchUser, bool isAutoProvisioned)
        {
            // Check auth
            var authResp = AuthenticationProvider.ProvisionMappedUser(int.Parse(twitchUser.UserID), isAutoProvisioned);

            if (authResp == null || (authResp.Status == ProvisionMappedUserStatus.Successful && authResp.MappedUser == null))
            {
                DiagLogger.Debug("No AuthResponse in ProvisionMappedUser", new { TwitchID = twitchUser.UserID, TwitchName = twitchUser.Username, authResponse = authResp, isAutoProvisioned = isAutoProvisioned });
                return new IdentityMergeState(IdentityMergeFailureReason.GeneralError);
            }
            
            if (authResp.Status != ProvisionMappedUserStatus.Successful)
            {
                DiagLogger.Debug("Provision request was not successful", new { TwitchID = twitchUser.UserID, TwitchName = twitchUser.Username, authResp, isAutoProvisioned });
                return new IdentityMergeState(GetFailureReason(authResp.Status));
            }

            // Provision Successful
            var status = authResp.MappedUser.Mode == EUserMapMode.AutoProvisioned ? IdentityMergeStatus.AutoProvisioned : IdentityMergeStatus.Merged;
            MergeLocalAccount(twitchUser, authResp.MappedUser.UserID, status == IdentityMergeStatus.AutoProvisioned);
            return new IdentityMergeState(twitchUser.UserID, authResp.MappedUser.UserID, twitchUser.Username, twitchUser.DisplayName, status);
        }

        public static IdentityMergeState MergeAccount(TwitchUserInfo twitchUser, string usernameOrEmail, string password)
        {
            // Check auth
            var authResp = AuthenticationProvider.MergeUserAccount(usernameOrEmail, password, int.Parse(twitchUser.UserID));
            
            // If not successful, return
            if (authResp.Status != MergeUserAccountStatus.Success)
            {
                return new IdentityMergeState(GetFailureReason(authResp.Status));
            }

            // If successful, setup local DB
            MergeLocalAccount(twitchUser, authResp.MappedUser.UserID);
            return new IdentityMergeState(twitchUser.UserID, authResp.MappedUser.UserID, twitchUser.Username, twitchUser.DisplayName);
        }

        public static IdentityMergeState MergeTempAccount(TempAccount tempAccount, TwitchUserInfo twitchUser)
        {
            // Check auth
            var authResp = AuthenticationProvider.MergeTempUserAccount(tempAccount.UserID, int.Parse(twitchUser.UserID));

            // If not successful, return
            if (authResp.Status != MergeUserAccountStatus.Success)
            {
                return new IdentityMergeState(GetFailureReason(authResp.Status));
            }

            // If successful, setup local DB
            MergeLocalAccount(twitchUser, authResp.MappedUser.UserID);            
            tempAccount.Status = TempAccountStatus.Claimed;
            tempAccount.Update();            
            return new IdentityMergeState(twitchUser.UserID, authResp.MappedUser.UserID, twitchUser.Username, twitchUser.DisplayName);
        }

        public static void UnmergeAccount(int twitchUserID)
        {
            var authResp = AuthenticationProvider.UnmergeUserAccount(twitchUserID);

            if (authResp.Status != UnmergeAccountStatus.Successful)
            {
                throw new Exception("Failed to unmerge from auth: " + authResp.Status);
            }

            // Upsert the ExternalAccount
            var externalAccount = ExternalAccount.GetByTwitchUserID(twitchUserID.ToString());
            UnmergeLocalUser(externalAccount, authResp.MappedUser.UserID, authResp.MappedUser.Mode == EUserMapMode.AutoProvisioned || authResp.MappedUser.Mode == EUserMapMode.Provisioned);
        }

        private static void UnmergeLocalUser(ExternalAccount account, int userID, bool forceDissolve = false)
        {
            // Fix local data
            account?.DeleteMapping(userID);
            var user = UserRegion.GetByUserID(userID)?.GetUser();
            if (user == null)
            {
                return;
            }

            if (forceDissolve || user.IsProvisional)
            {
                UserDissolveWorker.Create(user.UserID);
            }
            else
            {
                user.IsMerged = false;
                user.TwitchID = string.Empty;
                user.Update(p => p.TwitchID, p => p.IsMerged);
                user.UpdateStatistics(DateTime.UtcNow.ToEpochMilliseconds());
            }
        }

        public static void MergeLocalAccount(TwitchUserInfo twitchUser, int curseUserID, bool isAutoProvisioned = false)
        {
            // Upsert the UserRegion record      
            var userRegion = UserRegion.GetByUserID(curseUserID, false, false);

            if (userRegion == null)
            {
                userRegion = new UserRegion { UserID  = curseUserID, RegionID = UserRegion.LocalConfigID };
                userRegion.InsertLocal();
            }


            // Upsert the User record            
            var user = userRegion.GetUser();
            
            if (user == null)
            {
                user = new User
                {
                    UserID = curseUserID,
                    Username = twitchUser.Username,
                    DisplayName = twitchUser.DisplayName,
                    IsTempAccount = false,
                    GroupMessagePushPreference = PushNotificationPreference.Favorites,
                    FriendMessagePushPreference = PushNotificationPreference.All,
                    FriendRequestPushEnabled = true,
                    TwitchID = twitchUser.UserID,
                    IsMerged = true,
                    EmailAddress = twitchUser.Email,
                    EmailVerified = twitchUser.EmailVerified,
                    HasSyncedAccount = true,
                    IsProvisional = isAutoProvisioned,
                };

                user.Insert(userRegion.RegionID);                                          
            }
            else
            {
                // Don't update IsProvisional - that only applies to creation
                user.TwitchID = twitchUser.UserID;
                user.IsMerged = true;
                user.Username = twitchUser.Username;
                user.DisplayName = twitchUser.DisplayName;
                user.EmailAddress = twitchUser.Email;
                user.EmailVerified = twitchUser.EmailVerified;               
                user.IsTempAccount = false;
                user.HasSyncedAccount = true;
                user.IsProvisional = isAutoProvisioned;
                user.Update();
            }

            // Ensure stats are up to date, and set the token timestamp to a reasonable grace period
            user.UpdateStatistics(DateTime.UtcNow.AddSeconds(-30).ToEpochMilliseconds());

            if (!user.IsProvisional)
            {
                user.ReindexSearchHints();
            }

            UpdateExternalAccount(twitchUser, user, userRegion.RegionID, null, isAutoProvisioned);
            AccountRenameWorker.Create(user.UserID, user.Username, user.DisplayName, user.GetTitleName());

            if (!isAutoProvisioned)
            {
                TwitchMergeWorker.Create(user.UserID);
            }
        }

        private static void UpdateLocalAccount(TwitchUserInfo twitchUser, User user, UserRegion userRegion)
        {
            var nameChanged = user.Username != twitchUser.Username || user.DisplayName != twitchUser.DisplayName;
            user.Username = twitchUser.Username;
            user.DisplayName = twitchUser.DisplayName;
            user.EmailAddress = twitchUser.Email;
            user.EmailVerified = twitchUser.EmailVerified;
            user.Update();

            user.UpdateStatistics();

            if (!user.IsProvisional && nameChanged)
            {
                user.ReindexSearchHints();
            }
            UpdateExternalAccount(twitchUser, user, userRegion.RegionID, null, user.IsProvisional);

            if (nameChanged)
            {
                AccountRenameWorker.Create(user.UserID, user.Username, user.DisplayName, user.GetTitleName());
            }
        }
        
        private static void UpdateExternalAccount(TwitchUserInfo twitchUser, User curseUser, int regionID, ExternalAccount externalAccount = null, bool isAutoProvisioned = false)
        {
            // Upsert the ExternalAccount
            externalAccount = externalAccount ?? ExternalAccount.GetByTwitchUserID(twitchUser.UserID);
            var didExist = true;

            if (externalAccount == null)
            {
                didExist = false;
                externalAccount = new ExternalAccount();
                externalAccount.RegionID = regionID;
                externalAccount.Type = AccountType.Twitch;
                externalAccount.ExternalID = twitchUser.UserID;
                externalAccount.MappedUsers = new HashSet<int>();
            }

            if (!isAutoProvisioned)
            {
                externalAccount.AuthToken = twitchUser.AccessToken;
                externalAccount.RefreshToken = twitchUser.RefreshToken;
                externalAccount.Scopes = new HashSet<string>(twitchUser.Scopes ?? new List<string>());
                externalAccount.IsPartnered = twitchUser.IsPartner;
                externalAccount.ClientID = twitchUser.ClientID;
            }

            externalAccount.MergedUserID = curseUser.UserID;            
            externalAccount.AvatarUrl = twitchUser.Avatar;
            externalAccount.ExternalUsername = twitchUser.Username;
            externalAccount.ExternalDisplayName = twitchUser.DisplayName;
            externalAccount.LastSyncTime = DateTime.UtcNow;
            externalAccount.IsAutoProvisioned = isAutoProvisioned;

            var notifyReauth = externalAccount.NeedsReauthentication;
            externalAccount.NeedsReauthentication = false;

            if (didExist)
            {
                externalAccount.Update(a => a.AuthToken, a => a.RefreshToken, a => a.MergedUserID, a => a.AvatarUrl, a => a.ExternalUsername, a => a.ExternalDisplayName, a => a.IsPartnered,
                    a => a.Scopes, a => a.LastSyncTime, a => a.NeedsReauthentication, a=>a.IsAutoProvisioned, a=>a.ClientID);
            }
            else
            {
                externalAccount.InsertLocal();
            }

            // Check scopes to determine social ban status
            if (externalAccount.Scopes != null && externalAccount.Scopes.SetEquals(IdentityMergeConstants.BannedUserScopes))
            {
                UserBan.BanUser(curseUser.UserID, UserBanType.Social, false);
            }
            else
            {
                UserBan.UnbanUser(curseUser.UserID, UserBanType.Social, false);
            }

            // Ensure the community exists
            ExternalCommunity community = null;
            try
            {
                community = TwitchModelHelper.CreateOrUpdateCommunity(externalAccount.ExternalID, externalAccount, regionID);
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Failed to create a channel from Twitch account (account may be banned)");
            }


            // Map this user to the external account
            externalAccount.MapUser(curseUser.UserID, regionID, false, true);

            // Unmap all other users from this external account unless this is an auto-provision
            if (!isAutoProvisioned)
            {
                foreach (var otherUser in externalAccount.MappedUsers.Where(id => id != curseUser.UserID).ToArray())
                {
                    externalAccount.DeleteMapping(otherUser, true);
                }
            }

            // Unmap this user from other external accounts
            var myOtherLinks = ExternalAccountMapping.GetAllLocal(e => e.UserID, curseUser.UserID).Where(e => e.Type == AccountType.Twitch && e.ExternalID != externalAccount.ExternalID);
            var myOtherAccounts = ExternalAccount.MultiGetLocal(myOtherLinks.Select(l => new KeyInfo(l.ExternalID, AccountType.Twitch)));
            foreach (var otherAccount in myOtherAccounts)
            {
                otherAccount.DeleteMapping(curseUser.UserID, true);
            }

            if (notifyReauth && community != null && community.IsHostable)
            {
                ExternalCommunityCoordinator.OwnerReauthenticated(community, externalAccount);
            }

            Avatar.CreateOrUpdate(AvatarType.SyncedAccount, externalAccount.GetAvatarEntityID(), twitchUser.Avatar ?? string.Empty);
            curseUser.UpdateAvatarTimestamp(DateTime.UtcNow);
            if (string.IsNullOrEmpty(twitchUser.Avatar))
            {
                SyncAvatarToTwitchWorker.Create(curseUser.UserID, externalAccount.ExternalID);
            }
        }


        public static string CreateUsernameHmac(string username, string secretKey)
        {
            if (string.IsNullOrEmpty(username))
            {
                throw new ArgumentException("Username cannot be null or empty.");
            }

            return HashHmacHex(secretKey, username);
        }

        private static string HashHmacHex(string keyHex, string message)
        {
            var hash = HashHmac(HexDecode(keyHex), StringEncode(message));
            return HashEncode(hash);
        }


        private static string HashEncode(byte[] hash)
        {
            return BitConverter.ToString(hash).Replace("-", "").ToLower();
        }

        private static byte[] StringEncode(string text)
        {
            var encoding = new UTF8Encoding();
            return encoding.GetBytes(text);
        }

        private static byte[] HexDecode(string hex)
        {
            var bytes = new byte[hex.Length / 2];
            for (var i = 0; i < bytes.Length; i++)
            {
                bytes[i] = byte.Parse(hex.Substring(i * 2, 2), NumberStyles.HexNumber);
            }
            return bytes;
        }

        private static byte[] HashHmac(byte[] key, byte[] message)
        {
            var hash = new HMACSHA256(key);            
            return hash.ComputeHash(message);
        }

        /// <summary>
        /// Code should only be called after the users merge status has been verified. Update the users merge state and capture relevant oauth information at login which seems to either not be getting saved or later 
        /// dissapears after the very complicated merge user flow. 
        /// </summary>
        /// <param name="mergeState"></param>
        /// <param name="twitchUserInfo"></param>
        public static void UpdateMergedUserAccountInfo(IdentityMergeState mergeState, TwitchUserInfo twitchUserInfo)
        {
            if (mergeState == null || mergeState.Status != IdentityMergeStatus.Merged || twitchUserInfo == null)
            {
                ThrottledLogger.Debug("invalid or unmerged user logged in", new
                {
                    mergeState,
                    twitchUserInfoId = twitchUserInfo?.UserID
                });
                return;
            }

            var userRegion = UserRegion.GetByUserID(mergeState.CurseUserID);
            if (userRegion == null || userRegion.RegionID != UserRegion.LocalConfigID)
            {
                ThrottledLogger.Debug("User is logging in from a different region.", new
                {
                    curseUserId = mergeState.CurseUserID,
                    twitchUserId = mergeState.TwitchUserID
                });
                return;
            }
            
            // Update the user record with the correct merge state and twitch user id. 
            var user = userRegion.GetUser();
            if (user == null)
            {
                ThrottledLogger.Debug("User record not found in this region.");
                return;
            }

            if (!user.IsMerged)
            {
                ThrottledLogger.Debug("User record indicates not merged, but auth does.", new
                {
                    curseUserId = mergeState.CurseUserID, 
                    twitchUserId = mergeState.TwitchUserID
                });
                user.IsMerged = true;                    
                user.TwitchID = mergeState.TwitchUserID;
                user.Update(u => u.IsMerged, u => u.TwitchID);
            }                          

            // upsert external account with clientid and auth token. 
            var externalAccount = ExternalAccount.GetByTwitchUserID(twitchUserInfo.UserID);
                        
            // copied in part from UpdateMergedUserAccountInfo above but without some of the things I find a bit questionable.
            if (externalAccount == null)
            {
                ThrottledLogger.Debug("No External account for this user even though they're merged.", new
                {
                    twitchUserId = twitchUserInfo.UserID,
                });
                externalAccount = new ExternalAccount();
                externalAccount.RegionID = userRegion.RegionID;
                externalAccount.Type = AccountType.Twitch;
                externalAccount.ExternalID = twitchUserInfo.UserID;
                externalAccount.MappedUsers = new HashSet<int>();
                externalAccount.MergedUserID = mergeState.CurseUserID;
                externalAccount.AvatarUrl = twitchUserInfo.Avatar;
                externalAccount.ExternalUsername = twitchUserInfo.Username;
                externalAccount.ExternalDisplayName = twitchUserInfo.DisplayName;
                externalAccount.LastSyncTime = DateTime.UtcNow;
                externalAccount.IsAutoProvisioned = false;
                externalAccount.AuthToken = twitchUserInfo.AccessToken;
                externalAccount.RefreshToken = twitchUserInfo.RefreshToken;
                externalAccount.Scopes = new HashSet<string>(twitchUserInfo.Scopes ?? new List<string>());
                externalAccount.IsPartnered = twitchUserInfo.IsPartner;
                externalAccount.ClientID = twitchUserInfo.ClientID;

                externalAccount.InsertLocal();

                // Map this user to the external account
                externalAccount.MapUser(mergeState.CurseUserID, userRegion.RegionID, false, true);
                ThrottledLogger.Debug("re-created external account for merged user after oauth login.");
            }
            else
            {
                externalAccount.AuthToken = twitchUserInfo.AccessToken;
                externalAccount.RefreshToken = twitchUserInfo.RefreshToken;
                externalAccount.Scopes = new HashSet<string>(twitchUserInfo.Scopes ?? new List<string>());
                externalAccount.IsPartnered = twitchUserInfo.IsPartner;
                externalAccount.ClientID = twitchUserInfo.ClientID;
                externalAccount.Update(u => u.AuthToken, u => u.RefreshToken, u => u.Scopes, u => u.IsPartnered, u => u.ClientID, u => u.ClientID);
                ThrottledLogger.Debug("updated external account for user after oauth login.");
            }
        }
    }
}