﻿using System;
using System.Linq;
using Curse.Friends.UserEvents;
using Curse.Logging;
using Newtonsoft.Json;
using System.Collections.Generic;
using Curse.Friends.Tracing;
using Microsoft.CSharp.RuntimeBinder;

namespace Curse.Friends.TwitchInterop
{
    public abstract class BaseTwitchInteropContract
    {
        public abstract string Serialize();
        
        /// <summary>
        /// Hydrates the contract, and returns true if it should be relayed out.
        /// </summary>                
        public abstract bool ProcessOutbound(BaseUserEvent userEvent);

        /// <summary>
        /// Processes the Twitch interop contract, dispatching appropriate events.
        /// </summary>        
        public abstract void ProcessInbound();

        public abstract string GetContractName();

        public abstract bool Validate();

        public object GetLogData()
        {
            var obj = MemberwiseClone() as dynamic;
            try
            {
                obj.MessageBody = "<redacted>";
            }
            catch { }

            return obj;
        }

        public abstract void LogEvent(string message, object extraData = null);

        public abstract FilteredUserLogger FilteredLogger { get; }
    }

    public abstract class BaseTwitchInteropContract<TContract, TEvent> : BaseTwitchInteropContract 
        where TEvent : BaseUserEvent
        where TContract : BaseTwitchInteropContract
    {
        private static readonly string ContractName;
        protected static readonly LogCategory ValidationLogger;        
        protected static readonly LogCategory TraceLogger;
        protected static readonly LogCategory DiagLogger;
        protected static readonly FilteredUserLogger _filteredLogger;

        [JsonIgnore]
        public override FilteredUserLogger FilteredLogger
        {
            get { return _filteredLogger; }
        }
       
      
        protected void LogValidation(string message)
        {
            LogEvent(message);
        }

        public override string GetContractName()
        {
            return ContractName;
        }

        static BaseTwitchInteropContract()
        {
            var attribs = typeof (TContract).GetCustomAttributes(typeof (TwitchInteropContractAttribute), true).FirstOrDefault() as TwitchInteropContractAttribute;
            
            if (attribs == null)
            {
                throw new Exception("The type '" + typeof(TContract) + "' is missing a valid TwitchInteropContract attribute.");
            }

            ContractName = attribs.Name;
            ValidationLogger = new LogCategory(typeof(TContract).Name) { Throttle = TimeSpan.FromMinutes(1), ReleaseLevel = LogLevel.Trace };
            TraceLogger = new LogCategory(ContractName) { AlphaLevel = LogLevel.Trace };
            DiagLogger = new LogCategory(ContractName) { Throttle = TimeSpan.FromMinutes(1), ReleaseLevel = LogLevel.Debug };
            _filteredLogger = new FilteredUserLogger(ContractName);
        }


        public override void LogEvent(string message, object extraData = null)
        {
            var participantsContract = this as IParticipantsContract;
            if (participantsContract != null)
            {
                participantsContract.FilteredLogger.Log(participantsContract.ParticipantIDs, message, new { contract = participantsContract.GetLogData(), extraData });
                return;
            }

            var targetedContract = this as ITargetUserContract;
            if (targetedContract != null)
            {
                targetedContract.FilteredLogger.Log(new[] { targetedContract.TargetUserID, targetedContract.UserID }, message, new { contract = targetedContract.GetLogData(), extraData });
                return;
            }

            var userContract = this as IUserContract;
            if (userContract != null)
            {
                userContract.FilteredLogger.Log(userContract.UserID, message, new { contract = userContract.GetLogData(), extraData });
                return;
            }

            FilteredLogger.Log(null, null, message, new { contract = GetLogData(), extraData });
        }
        public override string Serialize()
        {            
            return JsonConvert.SerializeObject(this);
        }

        protected abstract bool ProcessOutbound(TEvent evt);
        
        public override bool ProcessOutbound(BaseUserEvent o)
        {
            var evt = o as TEvent;
            if (evt == null)
            {
                throw new ArgumentException("Supplied user event is not of the expected type.");
            }

            return ProcessOutbound(evt);
        }             
    }
}
