﻿using Curse.Friends.Data;
using Curse.Friends.Data.DerivedModels;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Curse.Friends.Enums;
using Curse.Friends.TwitchIdentityMerge;

namespace Curse.Friends.TwitchInterop
{
    /// <summary>
    /// Provider for ExternalAccount information
    /// </summary>
    public class ExternalAccountProvider
    {
        public ExternalAccount ExternalAccount { get; private set; }        
        public ExternalAccountProvider(string twitchID)
        {
            ExternalAccount = ExternalAccount.GetByTwitchUserID(twitchID);
        }
      
        public bool IsRequestorMerged()
        {
            return (ExternalAccount != null && ExternalAccount.MergedUserID > 0);             
        }

        public virtual bool Validate(out string error)
        {
            error = "";
            if(!IsRequestorMerged())
            {
                error = "account not found or merged.";
                return false;
            }
            return true;
        }                            
    }

    /// <summary>
    /// Provides both external and internal account information
    /// </summary>
    public class MergedAccountProvider : ExternalAccountProvider
    {
        public UserRegion Region { get; set; }
        public User InternalAccount { get; set; }
        public MergedAccountProvider(string twitchID)
            :base(twitchID)
        {
            // validate will take care of actual error reporting
            if (ExternalAccount != null && ExternalAccount.MergedUserID > 0)
            {
                Region = UserRegion.GetLocal(ExternalAccount.MergedUserID);
                InternalAccount = User.Get(Region.RegionID, ExternalAccount.MergedUserID);
            }
        }

        public override bool Validate(out string error)
        {
            if(!base.Validate(out error))
            {
                return false;
            }

            if(Region == null)
            {
                error = "User region could not be found.";
                return false;
            }

            if (InternalAccount == null)
            {
                error = "Internal account could not be found.";
                return false;
            }

            return true;
        }
    }    

    /// <summary>
    /// Provides friendship access for external accounts. 
    /// </summary>
    public class FriendshipContextProvider
    {
        public ExternalAccountProvider RequestorAccountProvider { get; private set; }
        public ExternalAccountProvider TargetAccountProvider { get; private set; }

        public FriendshipContext Context { get; private set; }
        public Friendship RequestorFriendship { get; private set; }
        public Friendship TargetFriendship { get; private set; }

        public FriendshipContextProvider(string requestorTwitchID, string targetTwitchID, bool autoProvision = false)            
        {
            RequestorAccountProvider = new ExternalAccountProvider(requestorTwitchID);
            TargetAccountProvider = new ExternalAccountProvider(targetTwitchID);

            var requestorHasAccount = RequestorAccountProvider.ExternalAccount != null && RequestorAccountProvider.ExternalAccount.IsMerged;
            var targetHasAccount = TargetAccountProvider.ExternalAccount != null && TargetAccountProvider.ExternalAccount.IsMerged;

            var requestorIsNotAutoProvisioned = requestorHasAccount && !RequestorAccountProvider.ExternalAccount.IsAutoProvisioned;
            var targetIsNotAutoProvisioned = targetHasAccount && !TargetAccountProvider.ExternalAccount.IsAutoProvisioned;

            var shouldProvision = autoProvision && (requestorIsNotAutoProvisioned || targetIsNotAutoProvisioned);

            if (shouldProvision && !RequestorAccountProvider.IsRequestorMerged())
            {
                var mergeState = IdentityMergeAuthHelper.ProvisionOrGetMappedUser(requestorTwitchID, true);
                if (mergeState.Status == IdentityMergeStatus.Merged || mergeState.Status == IdentityMergeStatus.AutoProvisioned)
                {
                    RequestorAccountProvider = new ExternalAccountProvider(requestorTwitchID);
                    requestorHasAccount = RequestorAccountProvider.ExternalAccount != null && RequestorAccountProvider.ExternalAccount.IsMerged;
                }
            }

            if (shouldProvision && !TargetAccountProvider.IsRequestorMerged())
            {
                var mergeState = IdentityMergeAuthHelper.ProvisionOrGetMappedUser(targetTwitchID, true);
                if (mergeState.Status == IdentityMergeStatus.Merged || mergeState.Status == IdentityMergeStatus.AutoProvisioned)
                {
                    TargetAccountProvider = new ExternalAccountProvider(targetTwitchID);
                    targetHasAccount = TargetAccountProvider.ExternalAccount != null && TargetAccountProvider.ExternalAccount.IsMerged;
                }
            }

            // don't want to throw exceptions, which context will do if it's given bs id's the validate call will handle error response.
            if (requestorHasAccount && targetHasAccount)
            {
                Context = new FriendshipContext(RequestorAccountProvider.ExternalAccount.MergedUserID, TargetAccountProvider.ExternalAccount.MergedUserID);
                RequestorFriendship = Context.GetMyFriendship();
                TargetFriendship = Context.GetTheirFriendship();
            }
        }

        private bool ValidateAccounts(out string error)
        {
            if (!RequestorAccountProvider.Validate(out error))
            {
                return false;
            }

            if (!TargetAccountProvider.Validate(out error))
            {
                return false;
            }
            return true;
        }

        private bool ValidateFriendship(out string error)
        {
            error = "";            
            if (RequestorFriendship == null && TargetFriendship == null)
            {
                error = "there are no friendships for these users.";
                return false;
            }

            if (RequestorFriendship == null || RequestorFriendship.Status == Enums.FriendshipStatus.Deleted)
            {
                error = "the requestor friendship is not found or is deleted.";
                return false;
            }

            if(TargetFriendship == null || TargetFriendship.Status == Enums.FriendshipStatus.Deleted)
            {
                error = "the target friendship is not found or is deleted.";
                return false;
            }
            return true;
        }

        public bool Validate(out string error, bool noFriendshipsExpected = false)
        {
            if (noFriendshipsExpected)
            {
                return ValidateAccounts(out error);
            }
            else
            {
                if(!ValidateAccounts(out error))
                {
                    return false; 
                }

                return ValidateFriendship(out error);
            }
        }

        public void DeleteOrphanedFriendships()
        {
            if ((RequestorFriendship == null || RequestorFriendship.Status == Enums.FriendshipStatus.Deleted) && (TargetFriendship != null && TargetFriendship.Status != Enums.FriendshipStatus.Deleted))
            {
                TargetFriendship.Status = Enums.FriendshipStatus.Deleted;
                TargetFriendship.Update(x => x.Status);                
            }
            else if ((TargetFriendship == null || TargetFriendship.Status == Enums.FriendshipStatus.Deleted) && (RequestorFriendship != null && RequestorFriendship.Status != Enums.FriendshipStatus.Deleted))
            {
                RequestorFriendship.Status = Enums.FriendshipStatus.Deleted;
                RequestorFriendship.Update(x => x.Status);                
            }
        }        
    }
}
