﻿using Curse.Friends.Data;
using Curse.Friends.Enums;
using System;
using System.Collections.Generic;
using System.Linq.Expressions;
using Curse.Friends.Tracing;
using Curse.Logging;

namespace Curse.Friends.TwitchInterop
{
    public class TwitchClientEndpointProvider
    {
        private static readonly Guid MachineKey = new Guid("fda79511-6dee-4b31-bc56-1e2983a9850a");
        
        public ClientEndpoint Endpoint { get; }

        public TwitchClientEndpointProvider(int userID)
        {
            Endpoint = ClientEndpoint.GetLocal(userID, MachineKey.ToString());

            if (Endpoint != null)
            {
                return;
            }

            Endpoint = ClientEndpoint.Create(userID, MachineKey.ToString(), MachineKey.ToString(), DevicePlatform.Twitch, string.Empty);
            Endpoint.ClientVersion = "7.0.0.0";
            Endpoint.Update(e => e.ClientVersion);
        }

        private static readonly FilteredUserLogger FilteredLogger = new FilteredUserLogger("TwitchEndpointProvider");

        public void UpdateSettings(UserRegion userRegion, UserConnectionStatus newOverride)
        {
            UserPresenceResolver.CreateAvailabilityOverrideResolver(userRegion.UserID, userRegion.RegionID, Endpoint.MachineKey, newOverride);
        }

        public void UpdateStatus(User user, UserRegion region, UserConnectionStatus newStatus, List<Expression<Func<ClientEndpoint,object>>> additionalUpdates = null)
        {
            var updates = additionalUpdates ?? new List<Expression<Func<ClientEndpoint, object>>>();

            switch (newStatus)
            {
                case UserConnectionStatus.Online:
                    updates.AddRange(CheckUpdates(Endpoint, true, false, DateTime.UtcNow, null));
                    break;
                case UserConnectionStatus.Offline:
                    updates.AddRange(CheckUpdates(Endpoint, false, false, null, DateTime.UtcNow));
                    break;
                case UserConnectionStatus.Idle:
                    updates.AddRange(CheckUpdates(Endpoint, true, true, null, null));
                    break;
                default:
                    // Error
                    Logger.Warn("Invalid/unsupported connection status: " + newStatus, user.UserID);
                    return;
            }

            if(updates.Count > 0)
            {
                Endpoint.Update(updates.ToArray());
                UserPresenceResolver.CreateAvailabilityResolver(user.UserID, region.RegionID, Endpoint.MachineKey);
            }

        }

        public static List<Expression<Func<ClientEndpoint, object>>> CheckUpdates(ClientEndpoint ep, bool isConnected, bool isIdle, DateTime? connectedDate, DateTime? disconnectedDate)
        {
            var changed = new List<Expression<Func<ClientEndpoint, object>>>();

            if (ep.IsConnected != isConnected)
            {
                ep.IsConnected = isConnected;
                changed.Add(e => e.IsConnected);
            }

            if (ep.IsIdle != isIdle)
            {
                ep.IsIdle = isIdle;
                changed.Add(e => e.IsIdle);
            }

            if (connectedDate.HasValue && ep.ConnectedDate != connectedDate.Value)
            {
                ep.ConnectedDate = connectedDate.Value;
                changed.Add(e => e.ConnectedDate);
            }

            if (disconnectedDate.HasValue && ep.DisconnectedDate != disconnectedDate.Value)
            {
                ep.DisconnectedDate = disconnectedDate.Value;
                changed.Add(e => e.DisconnectedDate);
            }

            return changed;
        }
    }
}
