﻿using Curse.Friends.UserEvents;
using Curse.Friends.Data;
using Curse.Friends.Statistics;
using Newtonsoft.Json;

namespace Curse.Friends.TwitchInterop.V1
{
    [TwitchInteropContract("cancel_friend_request")]
    public class CancelFriendRequestEventContract : BaseTwitchInteropContract<CancelFriendRequestEventContract, CancelFriendRequestEvent>, IUserContract, ITargetUserContract
    {
        [JsonProperty("user_id")]
        public string UserID { get; set; }

        [JsonProperty("target_user_id")]
        public string TargetUserID { get; set; }

        protected override bool ProcessOutbound(CancelFriendRequestEvent evt)
        {
            var pair = evt.GetUserPair();
            if (!pair.HaveTwitchIDs)
            {
                return false;
            }

            UserID = pair.User.TwitchID;
            TargetUserID = pair.OtherUser.TwitchID;
            return true;
        }

        public override void ProcessInbound()
        {           
            // UserID has sent a friend request that is still pending but now doesn't want to be friends with the target anymore... how sad.
            var provider = new FriendshipContextProvider(UserID, TargetUserID);

            var errorMessage = "";
            if (!provider.Validate(out errorMessage))
            {
                provider.DeleteOrphanedFriendships();
                LogValidation(string.Format("Supressing: {0}", errorMessage));
                return;
            }        

            if (provider.RequestorFriendship.Status == Enums.FriendshipStatus.AwaitingThem)
            {
                Friendship.Remove(false, provider.Context.Me, provider.RequestorFriendship, provider.Context.MyRegion, provider.Context.Them, provider.TargetFriendship, provider.Context.TheirRegion);

                // notifications to connected curse clients. 
                new FriendshipRemovedResolver { UserID = provider.Context.Me.UserID, FriendID = provider.Context.Them.UserID }.Enqueue();

                new FriendshipRemovedResolver { UserID = provider.Context.Them.UserID, FriendID = provider.Context.Me.UserID }.Enqueue();

                FriendsStatsManager.Current.FriendRequestsRemoved.Track();
            }
        }

        public override bool Validate()
        {
            if (string.IsNullOrWhiteSpace(UserID))
            {
                LogValidation("Supressing: No UserID");
                return false;
            }
            if (string.IsNullOrWhiteSpace(TargetUserID))
            {
                LogValidation("Supressing: No TargetID");
                return false;
            }
            if (UserID == TargetUserID)
            {
                LogValidation("Supressing: UserID and TargetID cannot match.");
                return false;
            }
            return true;
        }
    }
}
