﻿using Curse.Friends.UserEvents;
using Curse.Friends.Data;
using Curse.Friends.Statistics;
using Newtonsoft.Json;

namespace Curse.Friends.TwitchInterop.V1
{
    [TwitchInteropContract("accept_friend_request")]
    public class ConfirmFriendRequestEventContract : BaseTwitchInteropContract<ConfirmFriendRequestEventContract, ConfirmFriendRequestEvent>, IUserContract, ITargetUserContract
    {
        [JsonProperty("user_id")]
        public string UserID { get; set; }

        [JsonProperty("target_user_id")]
        public string TargetUserID { get; set; }

        protected override bool ProcessOutbound(ConfirmFriendRequestEvent evt)
        {
            var pair = evt.GetUserPair();
            if (!pair.HaveTwitchIDs)
            {
                return false;
            }
            UserID = pair.User.TwitchID;
            TargetUserID = pair.OtherUser.TwitchID;            
            return true;
        }

        public override void ProcessInbound()
        {
            // userid is accepting target's pending friend request.
            var provider = new FriendshipContextProvider(UserID, TargetUserID, true);
            string errorMessage = "";
            if(!provider.Validate(out errorMessage))
            {                
                provider.DeleteOrphanedFriendships();
                LogValidation(string.Format("Supressing: {0}", errorMessage));
                return;
            }                     
                        
            if (provider.RequestorFriendship.Status == Enums.FriendshipStatus.AwaitingMe)
            {
                Friendship.Confirm(provider.Context.Me, provider.Context.MyRegion, provider.RequestorFriendship, provider.Context.Them, provider.Context.TheirRegion, provider.TargetFriendship);
                FriendsStatsManager.Current.FriendRequestsConfirmed.Track();
            }
            else
            {
                LogValidation("The friendship between target and user is no longer awaiting the requestor.");
            }            
        }

        public override bool Validate()
        {
            if (string.IsNullOrWhiteSpace(UserID))
            {
                LogValidation("Supressing: No UserID");
                return false;
            }
            if (string.IsNullOrWhiteSpace(TargetUserID))
            {
                LogValidation("Supressing: No TargetID");
                return false;
            }
            if (UserID == TargetUserID)
            {
                LogValidation("Supressing: UserID and TargetID cannot match.");
                return false;
            }
            return true;
        }
    }
}
