﻿using System.Linq;
using Curse.Friends.TwitchIdentityMerge;
using Curse.Friends.UserEvents;
using Newtonsoft.Json;
using Curse.Friends.Data;
using Curse.Friends.Data.Queues;
using Curse.Logging;
using System;
using Curse.Extensions;
using Curse.Friends.Data.Messaging;
using System.Collections.Generic;

namespace Curse.Friends.TwitchInterop.V1
{
    [TwitchInteropContract("send_whisper")]
    public class ConversationMessageEventContract : BaseTwitchInteropContract<ConversationMessageEventContract, ConversationMessageEvent>, IUserContract, IParticipantsContract
    {
        [JsonProperty("message_id")]
        public Guid MessageID { get; set; }

        [JsonProperty("user_id")]
        public string UserID { get; set; }

        [JsonProperty("user_ip")]
        public string UserIpAddress { get; set; }

        [JsonProperty("unsanitized_body")]
        public string MessageBody { get; set; }

        [JsonProperty("participants")]
        public string[] ParticipantIDs { get; set; }

        [JsonProperty("message_timestamp")]
        public DateTime MessageTimestamp { get; set; }

        [JsonProperty("attached_file_url")]
        public string AttachmentUrl { get; set; }

        [JsonProperty("emoticons")]
        public EmoticonSubstitutionContract[] Emotes { get; set; }

        protected override bool ProcessOutbound(ConversationMessageEvent evt)
        {
            var pair = evt.GetSenderAndRecipient();

            if (!pair.HaveTwitchIDs)
            {
                return false;
            }


            MessageID = evt.MessageID;
            UserID = pair.User.TwitchID;
            UserIpAddress = evt.SenderIpAddress;
            MessageBody = evt.MessageBody;
            ParticipantIDs = pair.ParticipantTwitchIDs;
            MessageTimestamp = evt.MessageTimestamp.NormalizeToUtc();
            AttachmentUrl = evt.AttachmentUrl;
            return true;
        }
        
        public override void ProcessInbound()
        {
            
            var whisperInfo = WhisperHelper.GetWhisperInfo(UserID, ParticipantIDs, true);
            if (!whisperInfo.IsValid)
            {
                LogEvent("Supressed, due to failing validation or merge state", new { validationMessage = whisperInfo.ValidationMessage, validationData = whisperInfo.ValidationData});                
                return;
            }
            
            LogEvent("Sending whisper...");

            ExternalPrivateMessageWorker.Create(whisperInfo.UserID, UserIpAddress, whisperInfo.ConversationID, MessageBody, MessageID, MessageTimestamp.NormalizeToUtc(), whisperInfo.GetEmoteSubstitutionOverrides(Emotes, MessageBody));            
        }

        public override bool Validate()
        {
            if (MessageID == Guid.Empty)
            {
                LogValidation("Supressing: No MessageID");
                return false;
            }

            if (string.IsNullOrWhiteSpace(UserID))
            {
                LogValidation("Supressing: No UserID");
                return false;
            }

            if (string.IsNullOrWhiteSpace(UserIpAddress))
            {
                LogValidation("Supressing: No UserIPAddress");                     
                return false; 
            }

            if (string.IsNullOrWhiteSpace(MessageBody))
            {
                LogValidation("Supressing: No OriginalBody");
                return false;
            }

            if (ParticipantIDs == null || ParticipantIDs.Length != 2)
            {
                LogValidation("Supressing: Invalid Participant IDs");
                return false;
            }

            if(ParticipantIDs.Any(x => x == null))
            {
                LogValidation("Supressing: Null participants found.");
                return false;
            }

            if (ParticipantIDs[0] == ParticipantIDs[1])
            {
                LogValidation("Supressing: ParticipantIDs match");
                return false;
            }

            if (MessageTimestamp == DateTime.MinValue)
            {
                LogValidation("Supressing: No MessageTimestamp");
                return false;
            }

            return true;
        }
    }
}
