﻿using Curse.Friends.UserEvents;
using Newtonsoft.Json;
using System;
using System.Linq;
using Curse.Friends.Data.Messaging;
using Curse.Extensions;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.Data.Queues;

namespace Curse.Friends.TwitchInterop.V1
{
    [TwitchInteropContract("delete_whisper")]
    public class DeleteConversationMessageEventContract : BaseTwitchInteropContract<DeleteConversationMessageEventContract, DeleteConversationMessageEvent>, IUserContract, IParticipantsContract
    {
        [JsonProperty("user_id")]
        public string UserID { get; set; }       

        [JsonProperty("message_id")]
        public string MessageID { get; set; }

        [JsonProperty("participants")]
        public string[] ParticipantIDs { get; set; }

        public override void ProcessInbound()
        {
            var whisperInfo = WhisperHelper.GetWhisperInfo(UserID, ParticipantIDs, false);
            if (!whisperInfo.IsValid)
            {
                LogEvent("Supressing due to validation or merge state.", whisperInfo.ValidationData);                
                return;
            }

            // Get the container for the conversation (Group, Friend, etc)
            var conversationContainer = ConversationManager.GetConversationContainer(whisperInfo.UserID, whisperInfo.ConversationID);

            if (conversationContainer == null)
            {
                ValidationLogger.Debug("Supressing due to missing conversation container.");
                return;
            }

            var message = ConversationManager.FindMessageByID(whisperInfo.ConversationID, MessageID);

            if (message == null)
            {
                LogValidation("Supressing, message not found.");
                return;
            }

            
            var user = whisperInfo.GetUser();
            if (user == null)
            {
                return;
            }

            var editTimestamp = DateTime.UtcNow;

            // Make the change in memory, and dispatch it
            message.DeletedTimestamp = editTimestamp.ToEpochMilliseconds();
            message.DeletedUsername = user.GetTitleName();
            message.DeletedUserID = user.UserID;
            message.IsDeleted = true;
            conversationContainer.OnChatMessageChanged(message, ConversationNotificationType.Deleted);

            // Queue off a worker item to persist the change
            ConversationMessageWorker.CreateDeleteMessage(message.ConversationID, message.ID, editTimestamp, user.UserID, user.GetTitleName(), editTimestamp);
        }

        protected override bool ProcessOutbound(DeleteConversationMessageEvent evt)
        {
            var pair = evt.GetSenderAndRecipient();

            if (!pair.HaveTwitchIDs)
            {
                return false;
            }
            
            UserID = pair.User.TwitchID;
            MessageID = evt.MessageID;            
            return true;
        }

        public override bool Validate()
        {
            if (string.IsNullOrWhiteSpace(MessageID))
            {
                LogValidation("Supressing: No MessageID");
                return false;
            }

            if (string.IsNullOrWhiteSpace(UserID))
            {
                LogValidation("Supressing: No UserID");
                return false;
            }         

            if (ParticipantIDs == null || ParticipantIDs.Length != 2)
            {
                LogValidation("Supressing: Invalid Participant IDs");
                return false;
            }

            if (ParticipantIDs.Any(x => x == null))
            {
                LogValidation("Supressing: Null participants found.");
                return false;
            }

            if (ParticipantIDs[0] == ParticipantIDs[1])
            {
                LogValidation("Supressing: ParticipantIDs match");
                return false;
            }
            return true;
        }
    }
}
