﻿using System;
using Curse.Extensions;
using Curse.Friends.Data;
using Curse.Friends.Data.Messaging;
using Curse.Friends.UserEvents;
using Newtonsoft.Json;
using Curse.Friends.Enums;
using Curse.Friends.Data.Queues;
using System.Linq;
using Curse.Friends.Data.Models;

namespace Curse.Friends.TwitchInterop.V1
{
    [TwitchInteropContract("edit_whisper")]
    public class EditConversationMessageEventContract : BaseTwitchInteropContract<EditConversationMessageEventContract, EditConversationMessageEvent>, IUserContract, IParticipantsContract
    {
        [JsonProperty("user_id")]
        public string UserID { get; set; }

        [JsonProperty("message_id")]
        public string MessageID { get; set; }

        [JsonProperty("unsanitized_body")]
        public string MessageBody { get; set; }

        [JsonProperty("participants")]
        public string[] ParticipantIDs { get; set; }

        [JsonProperty("emoticons")]
        public EmoticonSubstitutionContract[] Emotes { get; set; }

        public override void ProcessInbound()
        {
            var whisperInfo = WhisperHelper.GetWhisperInfo(UserID, ParticipantIDs, false);
            if (!whisperInfo.IsValid)
            {
                LogEvent("Supressing due to validation or merge state.", whisperInfo.ValidationData);
                return;
            }

            var message = ConversationManager.FindMessageByID(whisperInfo.ConversationID, MessageID);

            if (message == null)
            {
                LogValidation("Supressing, message not found.");
                return;
            }

            var user = User.FindByUserID(whisperInfo.UserID);

            if (user == null)
            {
                LogValidation("Supressing, user not found.");
                return;
            }

            var conversationContainer = ConversationManager.GetConversationContainer(user.UserID, message.ConversationID);

            if (!conversationContainer.CanEditMessage(user.UserID, message))
            {
                LogValidation("Supressing, no permission to edit message.");
                return;
            }

            var editTimestamp = DateTime.UtcNow;

            // Make the change in memory, and dispatch it
            message.EditMessage(user.UserID, user.GetTitleName(), MessageBody, editTimestamp.ToEpochMilliseconds(), null, whisperInfo.GetEmoteSubstitutionOverrides(Emotes, MessageBody));

            conversationContainer.OnChatMessageChanged(message, ConversationNotificationType.Edited);

            // Queue off a worker item to persist the change
            ConversationMessageWorker.CreateEditMessage(message.ConversationID, message.ID, editTimestamp, user.UserID, user.GetTitleName(), MessageBody, editTimestamp, null, message.EmoteSubstitutions);

        }

        protected override bool ProcessOutbound(EditConversationMessageEvent evt)
        {
            var pair = evt.GetSenderAndRecipient();

            if (!pair.HaveTwitchIDs)
            {
                return false;
            }


            UserID = pair.User.TwitchID;
            MessageID = evt.MessageID;
            MessageBody = evt.MessageBody;
            return true;
        }

        public override bool Validate()
        {
            if (string.IsNullOrWhiteSpace(MessageID))
            {
                LogValidation("Supressing: No MessageID");
                return false;
            }

            if (string.IsNullOrWhiteSpace(UserID))
            {
                LogValidation("Supressing: No UserID");
                return false;
            }

            if (ParticipantIDs == null || ParticipantIDs.Length != 2)
            {
                LogValidation("Supressing: Invalid Participant IDs");
                return false;
            }

            if (ParticipantIDs.Any(x => x == null))
            {
                LogValidation("Supressing: Null participants found.");
                return false;
            }

            if (ParticipantIDs[0] == ParticipantIDs[1])
            {
                LogValidation("Supressing: ParticipantIDs match");
                return false;
            }

            if (string.IsNullOrWhiteSpace(MessageBody))
            {
                LogValidation("Supressing: No MessageBody");
                return false;
            }

            return true;
        }
    }
}
