﻿using System;
using Curse.Friends.UserEvents;
using Newtonsoft.Json;
using System.Linq; 

namespace Curse.Friends.TwitchInterop.V1
{
    [TwitchInteropContract("archive_whisper")]
    public class HideConversationEventContract : BaseTwitchInteropContract<HideConversationEventContract, HideConversationEvent>, IUserContract, IParticipantsContract
    {
        [JsonProperty("user_id")]
        public string UserID { get; set; }

        [JsonProperty("participants")]
        public string[] ParticipantIDs { get; set; }

        [JsonProperty("is_archived")]
        public bool IsArchived { get; set; }

        protected override bool ProcessOutbound(HideConversationEvent evt)
        {
            var pair = evt.GetSenderAndRecipient();

            if (!pair.HaveTwitchIDs)
            {
                return false;
            }

            UserID = pair.User.TwitchID;
            ParticipantIDs = pair.ParticipantTwitchIDs;
            IsArchived = evt.IsHidden;
            return true;
        }

        public override void ProcessInbound()
        {
            var whisperInfo = WhisperHelper.GetWhisperInfo(UserID, ParticipantIDs, false);

            if (!whisperInfo.IsValid)
            {
                LogEvent("Supressing due to validation or merge state.", whisperInfo.ValidationData);
                return;
            }

            whisperInfo.ToggleHidden(IsArchived);
        }

        public override bool Validate()
        {
            if (string.IsNullOrWhiteSpace(UserID))
            {
                LogValidation("Supressing: No UserID");
                return false;
            }

            if (ParticipantIDs == null || ParticipantIDs.Length != 2)
            {
                LogValidation("Supressing: Invalid Participant IDs");
                return false;
            }

            if (ParticipantIDs.Any(x => x == null))
            {
                LogValidation("Supressing: Null participants found.");
                return false;
            }

            if (ParticipantIDs[0] == ParticipantIDs[1])
            {
                LogValidation("Supressing: ParticipantIDs match");
                return false;
            }

            return true;
        }
    }

}
