﻿using Curse.Friends.Data;
using Curse.Friends.Data.Models;
using Curse.Friends.Enums;
using Curse.Friends.UserEvents;
using Curse.Logging;
using Newtonsoft.Json;
using System;
using Curse.Friends.Data.Queues;

namespace Curse.Friends.TwitchInterop.V1
{
    [TwitchInteropContract("set_presence_settings")]
    public class PresenceSettingsEventContract : BaseTwitchInteropContract<PresenceSettingsEventContract, PresenceSettingsEvent>, IUserContract
    {
        [JsonProperty("user_id")]
        public string UserID { get; set; }
        [JsonProperty("is_invisible")]
        public bool IsInvisible { get; set; }
        [JsonProperty("share_activity")]
        public bool ShareActivity { get; set; }

        protected override bool ProcessOutbound(PresenceSettingsEvent evt)
        {
            var user = evt.GetUserStatistics();

            if (!user.HasTwitchID)
            {
                LogEvent("Suppressing outbound event. Account has not been merged.");
                return false;
            }

            UserID = user.TwitchID;
            IsInvisible = evt.IsInvisible;
            ShareActivity = evt.ShareActivity;

            FilteredLogger.Log(user.UserID, "Publishing presence settings.");
            return true;
        }

        public override void ProcessInbound()
        {
            var acctProvider = new MergedAccountProvider(UserID);

            string validationError;
            if (!acctProvider.Validate(out validationError))
            {
                LogValidation(validationError);                
                return;
            }

            var user = acctProvider.InternalAccount;
            var region = acctProvider.Region;

            // Share Activity
            try
            {
                var userPrivacy = user.GetPrivacy();
                var newActivityPrivacy = ShareActivity ? ShareActivityPrivacy.Share : ShareActivityPrivacy.Hide;
                if (userPrivacy.ShareActivityPrivacy != newActivityPrivacy)
                {
                    userPrivacy.ShareActivityPrivacy = newActivityPrivacy;
                    userPrivacy.Update(u => u.ShareActivityPrivacy);
                    user.UpdateStatistics();
                    RegionalUserChangeResolver.Create(user.UserID);
                }

                FilteredLogger.Log(user, "Received share activity privacy");
            }
            catch(Exception ex)
            {
                Logger.Warn(ex, "Failed to save user's share activity setting, skipping", new { Contract = this, CurseUserID = acctProvider.ExternalAccount.MergedUserID });
            }

            
            // Invisible
            if (IsInvisible)
            {
                if (user.ConnectionStatus != UserConnectionStatus.Invisible)
                {
                    var ep = new TwitchClientEndpointProvider(user.UserID).Endpoint;
                    UserPresenceResolver.CreateAvailabilityOverrideResolver(user.UserID, region.RegionID, ep.MachineKey, UserConnectionStatus.Invisible);
                    FilteredLogger.Log(user, "Resolving status to Invisible");
                }
                else
                {
                    FilteredLogger.Log(user, "Supressing Invisible status, as it is already set.");
                }
            }
            else
            {
                if(user.ConnectionStatus == UserConnectionStatus.Invisible)
                {
                    var ep = new TwitchClientEndpointProvider(user.UserID).Endpoint;
                    UserPresenceResolver.CreateAvailabilityOverrideResolver(user.UserID, region.RegionID, ep.MachineKey, UserConnectionStatus.Online);
                    FilteredLogger.Log(user, "Resolving status to Online");
                }               
            }

            FilteredLogger.Log(acctProvider.InternalAccount, "Processed inbound presence settings event.");
        }

        public override bool Validate()
        {
            if (string.IsNullOrWhiteSpace(UserID))
            {
                LogValidation("Supressing: No UserID");
                return false;
            }
            return true;
        }
    }
}
