﻿using System;
using System.Collections.Generic;
using System.Linq;
using Newtonsoft.Json;
using Curse.Friends.Data;
using Curse.Friends.UserEvents;
using Curse.Friends.Enums;
using Curse.Logging;
using System.Linq.Expressions;
using Curse.Friends.Tracing;

namespace Curse.Friends.TwitchInterop.V1
{
    [TwitchInteropContract("set_presence_status")]
    public class PresenceStatusEventContract : BaseTwitchInteropContract<PresenceStatusEventContract, PresenceStatusEvent>, IUserContract
    {
        [JsonProperty("user_id")]
        public string UserID { get; set; }

        [JsonProperty("availability")]
        public string Availability { get; set; }

        [JsonProperty("activity")]
        public PresenceActivity Activity { get; set; }
        
        public override void ProcessInbound()
        {
            FilteredLogger.Log(UserID, "Received inbound presence event", GetLogData());

            // Look up user by Twitch ID
            var acct = ExternalAccount.GetByTwitchUserID(UserID);

            // Not tracked
            if (acct == null)
            {
                FilteredLogger.Log(UserID, "Supressing inbound event. User does not exist locally.", this);
                return;
            }

            // Not merged
            if (acct.MergedUserID <= 0)
            {
                FilteredLogger.Log(null, acct, "Supressing inbound event. User is not merged.", this);                
                return;
            }

            // Look up client endpoint for Twitch Endpoint
            var region = UserRegion.GetByUserID(acct.MergedUserID);
            if (region == null)
            {
                // Should never happen, no region for a mapped user
                FilteredLogger.Warn(null, acct, "Failed to retrieve user region for merged account.", new { acct.MergedUserID });
                return;
            }

            var user = region.GetUser();

            if (user == null)
            {
                // Should never happen, no user record for a mapped user
                FilteredLogger.Warn(null, acct, "Failed to retrieve user for merged account.", new { acct.MergedUserID });
                return;
            }

            // See if this is an actual change            
            var newStatus = GetConnectionStatus(Availability);

            // Update client endpoint 
            var endpointProvider = new TwitchClientEndpointProvider(region.UserID);
            var endpoint = endpointProvider.Endpoint;

            // Activity
            var updates = CheckActivityUpdates(user, endpoint);

            if (!updates.Any() && newStatus != user.ConnectionStatus)
            {
                FilteredLogger.Log(user, acct, "Suppressing inbound event. No changes detected to rich presence or connection status.", new { user = user.GetLogData(), newStatus, contract = this });
                return;
            }
            
            if (!endpoint.IsConnected && newStatus != UserConnectionStatus.Offline)
            {
                endpoint.SessionDate = DateTime.UtcNow;
                updates.Add(e => e.SessionDate);
            }

            FilteredLogger.Log(user, acct, "Resolving user status to " + newStatus, new { user = user.GetLogData(), contract = this });
            endpointProvider.UpdateStatus(user, region, newStatus, updates);
        }

        private List<Expression<Func<ClientEndpoint, object>>> CheckActivityUpdates(User user, ClientEndpoint ep)
        {            
            var changed = new List<Expression<Func<ClientEndpoint, object>>>();
            if (Activity == null)
            {
                FilteredLogger.Warn(user, null, "Event did not include an Activity object");
                return changed;
            }

            var playingGameID = 0;
            var broadcastingGameID = 0;
            var isBroadcasting = false;

            var watchingChannelID = string.Empty;
            switch (Activity.Type)
            {
                case "watching":
                    watchingChannelID = Activity.ChannelID;
                    FilteredLogger.Log(user, null, "Received watching presence update");
                    break;
                case "playing":
                    playingGameID = GetGameIDFromTwitchGameID(Activity.GameID);
                    FilteredLogger.Log(user, null, "Received playing presence update");
                    break;
                case "broadcasting":
                    isBroadcasting = true;
                    if (!string.IsNullOrEmpty(Activity.GameID))
                    {
                        broadcastingGameID = GetGameIDFromTwitchGameID(Activity.GameID);
                    }
                    else if (!string.IsNullOrEmpty(Activity.GameTitle))
                    {
                        broadcastingGameID = GameMapping.GetCurseID(Activity.GameTitle);
                    }
                        
                    FilteredLogger.Log(user, null, "Received broadcasting presence update", new { user = user.GetLogData(), contract = this, broadcastingGameID });
                    break;
                case "none":
                    FilteredLogger.Log(user, null, "Received none presence update");
                    break;
                default:
                    FilteredLogger.Log(user, null, "Unknown activity type: " + Activity.Type);
                    break;                    
            }

            if (isBroadcasting != ep.IsBroadcasting)
            {
                ep.IsBroadcasting = isBroadcasting;
                changed.Add(p => p.IsBroadcasting);
            }
            
            if (broadcastingGameID != ep.CurrentlyBroadcastingGameID)
            {
                ep.CurrentlyBroadcastingGameID = broadcastingGameID;
                changed.Add(p => p.CurrentlyBroadcastingGameID);
            }

            if (watchingChannelID != ep.CurrentlyWatchingChannelID)
            {
                ep.CurrentlyWatchingChannelID = watchingChannelID;
                changed.Add(p => p.CurrentlyWatchingChannelID);
            }

            if (playingGameID != ep.CurrentlyPlayingGameID)
            {
                ep.CurrentlyPlayingGameID = playingGameID;
                ep.CurrentlyPlayingGameTimestamp = DateTime.UtcNow;
                changed.Add(p => p.CurrentlyPlayingGameID);
                changed.Add(p => p.CurrentlyPlayingGameTimestamp);
            }

            if (changed.Count > 0)
            {
                ep.ActivityTimestamp = DateTime.UtcNow;
                changed.Add(p => p.ActivityTimestamp);
            }
            return changed;
        }

        private static int GetGameIDFromTwitchGameID(string twitchGameID)
        {
            int id;
            if (int.TryParse(twitchGameID, out id))
            {
                return GameMapping.GetCurseID(id);
            }                        
            return 0;
        }        

        protected override bool ProcessOutbound(PresenceStatusEvent evt)
        {
            PresenceActivity activity = null;

            var user = evt.GetUser();

            if (!user.IsMerged)
            {
                LogEvent("Suppressing outbound event. Account has not been merged.", evt);
                return false;
            }

            if (evt.GameID > 0)
            {                
                activity = new PresenceActivity { Type = "playing", GameID = GameMapping.GetTwitchID(evt.GameID).ToString() };
            }
            else if (!string.IsNullOrEmpty(evt.ChannelID))
            {
                var channel = ExternalCommunity.GetByTwitchID(evt.ChannelID);
                if (channel != null)
                {
                    activity = new PresenceActivity { Type = "watching", ChannelID = evt.ChannelID.ToString(), ChannelDisplayName = channel.ExternalDisplayName, ChannelUrl = channel.GetExternalUrl() };
                }
            }

            activity = activity ?? new PresenceActivity { Type = "none" };


            UserID = user.TwitchID;
            Availability = GetConnectionStatusString(evt.ConnectionStatus);
            Activity = activity;

            FilteredLogger.Log(user, "Sending presence update", GetLogData());

            // If the user needs to sync their account with Twitch (friends, settings, etc)
            if (user.ShouldSyncSocially())
            {
                FilteredLogger.Log(user, "Syncing account with Twitch social", GetLogData());
                TwitchMergeWorker.Create(user.UserID);
            }
            
            return true;
        }

        private static UserConnectionStatus GetConnectionStatus(string status)
        {
            switch (status)
            {
                case "online":
                  return UserConnectionStatus.Online;
                case "offline":
                    return UserConnectionStatus.Offline;
                case "idle":
                    return UserConnectionStatus.Idle;
                case "away":
                    return UserConnectionStatus.Away;                    
                case "busy":
                    return UserConnectionStatus.DoNotDisturb;                    
                case "invisible":
                    return UserConnectionStatus.Invisible;                    
                default:
                    // Error
                    return UserConnectionStatus.Offline;
            }
        }

        private static string GetConnectionStatusString(UserConnectionStatus status)
        {
            switch (status)
            {
                case UserConnectionStatus.Online:
                    return "online";
                case UserConnectionStatus.Away:
                case UserConnectionStatus.Idle:
                    return "away";
                case UserConnectionStatus.DoNotDisturb:
                    return "busy";
                case UserConnectionStatus.Invisible:
                    return "invisible";
                case UserConnectionStatus.Offline:
                    return "offline";
                default:
                    return "offline";
            }
        }

        public override bool Validate()
        {
            if (string.IsNullOrWhiteSpace(UserID))
            {
                LogValidation("Supressing: No UserID");
                return false;
            }

            if (string.IsNullOrWhiteSpace(Availability))
            {
                LogValidation("Supressing: No Availability");
                return false;
            }
            return true;
        }
    }


    public class PresenceActivity
    {
        [JsonProperty("type")]
        public string Type { get; set; }

        [JsonProperty("channel_id")]
        public string ChannelID { get; set; }

        [JsonProperty("channel_display_name")]
        public string ChannelDisplayName { get; set; }

        [JsonProperty("channel_url")]
        public string ChannelUrl { get; set; }

        [JsonProperty("game_id")]
        public string GameID { get; set; }

        [JsonProperty("game")]
        public string GameTitle { get; set; }
    }
}
