﻿using System;
using Curse.Extensions;
using Curse.Friends.UserEvents;
using Newtonsoft.Json;
using System.Linq;
using Curse.Logging;
using Curse.Friends.Data;
using Curse.Friends.Data.Queues;

namespace Curse.Friends.TwitchInterop.V1
{
    [TwitchInteropContract("read_whisper")]
    public class ReadConversationEventContract : BaseTwitchInteropContract<ReadConversationEventContract, ReadConversationEvent>, IUserContract, IParticipantsContract
    {
        [JsonProperty("user_id")]
        public string UserID { get; set; }

        [JsonProperty("message_id")]
        public string MessageID { get; set; }

        [JsonProperty("message_timestamp")]
        public DateTime MessageTimestamp { get; set; }

        [JsonProperty("participants")]
        public string[] ParticipantIDs { get; set; }
     
        protected override bool ProcessOutbound(ReadConversationEvent evt)
        {
            var pair = evt.GetSenderAndRecipient();

            if (!pair.HaveTwitchIDs)
            {
                return false;
            }

            MessageID = evt.MessageID;
            UserID = pair.User.TwitchID;
            MessageTimestamp = evt.MessageTimestamp.FromEpochMilliconds();
            ParticipantIDs = pair.ParticipantTwitchIDs;

            return true;
        }

        public override void ProcessInbound()
        {                       
            var requestorAccountProvider = new ExternalAccountProvider(UserID);
            string error; 
            if (!requestorAccountProvider.Validate(out error))
            {
                LogValidation(string.Format("Supressing: {0}", error));
                return;
            }

            var otherID = ParticipantIDs.FirstOrDefault(x => x != UserID);
            var otherAccountProvider = new ExternalAccountProvider(otherID);
            if (!otherAccountProvider.Validate(out error))
            {
                LogValidation(string.Format("Supressing: {0}", error));
                return;
            }

            if (requestorAccountProvider.ExternalAccount.IsAutoProvisioned && otherAccountProvider.ExternalAccount.IsAutoProvisioned)
            {
                LogValidation("Both accounts are provisional, no conversation exists.");
                return;
            }

            var conversationID = PrivateConversation.GenerateConversationID(requestorAccountProvider.ExternalAccount.MergedUserID, otherAccountProvider.ExternalAccount.MergedUserID);           

            // Add 999 ms to deal with nanosecond rounding
            var utcDate = MessageTimestamp.NormalizeToUtc().AddMilliseconds(999);

            ConversationReadWorker.Create(requestorAccountProvider.ExternalAccount.MergedUserID, conversationID, utcDate, null, MessageID, true);            
        }


        public override bool Validate()
        {
            if (string.IsNullOrWhiteSpace(UserID))
            {
                LogValidation("No userid provided.");
                return false; 
            }
            if (string.IsNullOrWhiteSpace(MessageID))
            {
                LogValidation("No MessageID provided.");
                return false;                
            }
            if (MessageTimestamp == DateTime.MinValue)
            {
                LogValidation("No MessageTimestamp provided.");
                return false;             
            }
            if (ParticipantIDs == null || ParticipantIDs.Length != 2)
            {
                LogValidation("ParticipantIDs not provided.");
                return false;
            }
            if (ParticipantIDs.Any(x => x == null))
            {
                LogValidation("Supressing: Null participants found.");
                return false;
            }
            if (ParticipantIDs[0] == ParticipantIDs[1])
            {
                LogValidation("ParticipantIDs are the same.");
                return false;
            }
            return true; 
        }       
    }
}
