﻿using Curse.Friends.Data;
using Curse.Friends.UserEvents;
using Newtonsoft.Json;

namespace Curse.Friends.TwitchInterop.V1
{
    [TwitchInteropContract("remove_friend")]
    public class RemoveFriendshipEventContract : BaseTwitchInteropContract<RemoveFriendshipEventContract, RemoveFriendshipEvent>, IUserContract, ITargetUserContract
    {
        [JsonProperty("user_id")]
        public string UserID { get; set; }

        [JsonProperty("target_user_id")]
        public string TargetUserID { get; set; }

        protected override bool ProcessOutbound(RemoveFriendshipEvent evt)
        {
            var pair = evt.GetUserPair();
            if (!pair.HaveTwitchIDs)
            {
                return false;
            }
            UserID = pair.User.TwitchID;
            TargetUserID = pair.OtherUser.TwitchID;
            return true;
        }

        public override void ProcessInbound()
        {
            var provider = new FriendshipContextProvider(UserID, TargetUserID);
            var errorMessage = "";
            if (!provider.Validate(out errorMessage))
            {
                provider.DeleteOrphanedFriendships();
                LogValidation(string.Format("Supressing: {0}", errorMessage));
                return;
            }

            if (!provider.RequestorFriendship.IsSurfaced)
            {
                LogValidation("Supressing: Attempting to remove friendship that is not confirmed or pending.");
                return;
            }

            Friendship.Remove(false, provider.Context.Me, provider.RequestorFriendship, provider.Context.MyRegion, provider.Context.Them, provider.TargetFriendship, provider.Context.TheirRegion);

            // Notify Myself        
            new FriendshipRemovedResolver { UserID = provider.Context.Me.UserID, FriendID = provider.Context.Them.UserID }.Enqueue();

            // Notify Them
            new FriendshipRemovedResolver { UserID = provider.Context.Them.UserID, FriendID = provider.Context.Me.UserID }.Enqueue();
        }

        public override bool Validate()
        {
            if (string.IsNullOrWhiteSpace(UserID))
            {
                LogValidation("Supressing: No UserID");
                return false;
            }
            if (string.IsNullOrWhiteSpace(TargetUserID))
            {
                LogValidation("Supressing: No TargetID");
                return false;
            }
            if (UserID == TargetUserID)
            {
                LogValidation("Supressing: UserID and TargetID cannot match.");
                return false;
            }
            return true;
        }
    }
}
