﻿using Curse.Friends.UserEvents;
using Newtonsoft.Json;
using Curse.Friends.Data;
using Curse.Friends.Enums;

namespace Curse.Friends.TwitchInterop.V1
{
    [TwitchInteropContract("request_friend")]
    public class RequestFriendshipEventContract : BaseTwitchInteropContract<RequestFriendshipEventContract, RequestFriendshipEvent>, IUserContract, ITargetUserContract
    {
        [JsonProperty("user_id")]
        public string UserID { get; set; }

        [JsonProperty("target_user_id")]
        public string TargetUserID { get; set; }

        [JsonProperty("is_recommended")]
        public bool IsRecommended { get; set; }

        protected override bool ProcessOutbound(RequestFriendshipEvent evt)
        {
            var pair = evt.GetUserPair();
            if (!pair.HaveTwitchIDs)
            {
                return false;
            }
            UserID = pair.User.TwitchID;
            TargetUserID = pair.OtherUser.TwitchID;
            IsRecommended = evt.IsFromSuggestion;
            return true;
        }

        public override void ProcessInbound()
        {
            var provider = new FriendshipContextProvider(UserID, TargetUserID, true);
            var errorMessage = "";
            if (!provider.Validate(out errorMessage, true))
            {
                LogValidation(string.Format("Supressing: {0}", errorMessage));
                return;
            }


            if (provider.RequestorFriendship != null && provider.RequestorFriendship.Status != FriendshipStatus.Deleted)
            {
                // if a friendship is already confirmed, pending confirmation from the other person, or has been declined by the other person then do nothing.
                if (provider.RequestorFriendship.Status == FriendshipStatus.Confirmed ||
                    provider.RequestorFriendship.Status == FriendshipStatus.AwaitingThem ||
                    provider.RequestorFriendship.Status == FriendshipStatus.DeclinedByThem)
                {
                    return;
                }
                
                // if I'm requesting a friendship with someone that is waiting on my to approve a friend request then go ahead and approve. 
                if (provider.RequestorFriendship.Status == FriendshipStatus.AwaitingMe)
                {
                    Friendship.Confirm(provider.Context.Me, provider.Context.MyRegion, provider.RequestorFriendship, provider.Context.Them, provider.Context.TheirRegion, provider.TargetFriendship);
                    new ConfirmFriendRequestEvent { UserID = provider.Context.Me.UserID, OtherUserID = provider.Context.Them.UserID }.Enqueue();
                }
            }
            else
            {
                if(!User.CheckFriendshipPrivacy(provider.Context.Me, provider.Context.Them))
                {                    
                    return;
                }

                // no pending or confirmed friendship so create new requests for each person. 
                Friendship.CreateRequest(provider.Context.Me, provider.Context.MyRegion,
                    provider.Context.Them, provider.Context.TheirRegion, provider.Context.Them.Username,
                    null, // invitation message
                    IsRecommended, provider.RequestorFriendship, provider.TargetFriendship);
            }
        }

        public override bool Validate()
        {
            if (string.IsNullOrWhiteSpace(UserID))
            {
                LogValidation("Supressing: No UserID");
                return false;
            }
            if (string.IsNullOrWhiteSpace(TargetUserID))
            {
                LogValidation("Supressing: No TargetID");
                return false;
            }
            if (UserID == TargetUserID)
            {
                LogValidation("Supressing: UserID and TargetID cannot match.");
                return false;
            }
            return true;
        }
    }
}
