﻿using System;
using Curse.Friends.Data.Messaging;
using Curse.Friends.UserEvents;
using Newtonsoft.Json;
using System.Linq; 

namespace Curse.Friends.TwitchInterop.V1
{
    [TwitchInteropContract("unmute_whisper")]
    public class UnmuteConversationEventContract : BaseTwitchInteropContract<UnmuteConversationEventContract, UnmuteConversationEvent>
    {
        [JsonProperty("user_id")]
        public string UserID { get; set; }

        [JsonProperty("participants")]
        public string[] ParticipantIDs { get; set; }

        protected override bool ProcessOutbound(UnmuteConversationEvent evt)
        {
            var pair = evt.GetSenderAndRecipient();

            if (!pair.HaveTwitchIDs)
            {
                return false;
            }


            UserID = pair.User.TwitchID;
            ParticipantIDs = pair.ParticipantTwitchIDs;
            return false;
        }

        public override void ProcessInbound()
        {
            var whisperInfo = WhisperHelper.GetWhisperInfo(UserID, ParticipantIDs, false);
            
            if (!whisperInfo.IsValid)
            {
                ValidationLogger.Debug("Supressing due to validation or merge state.", whisperInfo.ValidationData);
                return;
            }

            whisperInfo.ToggleMuted(false);
        }

        public override bool Validate()
        {
            if (string.IsNullOrWhiteSpace(UserID))
            {
                LogValidation("Supressing: No UserID");
                return false;
            }

            if (ParticipantIDs == null || ParticipantIDs.Length != 2)
            {
                LogValidation("ParticipantIDs not provided.");
                return false;
            }

            if (ParticipantIDs.Any(x => x == null))
            {
                LogValidation("Supressing: Null participants found.");
                return false;
            }

            if (ParticipantIDs[0] == ParticipantIDs[1])
            {
                LogValidation("ParticipantIDs are the same.");
                return false;
            }

            return true;
        }
    }
}
