﻿using Curse.Friends.Data;
using Curse.Friends.Data.Models;
using Curse.Friends.Enums;
using Curse.Friends.UserEvents;
using Curse.Logging;
using Newtonsoft.Json;
using System;
using Curse.Friends.Data.Queues;

namespace Curse.Friends.TwitchInterop.V2
{
    [TwitchInteropContract("set_presence_settings_v2")]
    public class PresenceSettingsEventContractV2 : BaseTwitchInteropContract<PresenceSettingsEventContractV2, PresenceSettingEventV2>, IUserContract
    {
        [JsonProperty("user_id")]
        public string UserID { get; set; }
        [JsonProperty("availability_override")]
        public string AvailabilityOverride { get; set; }
        [JsonProperty("share_activity")]
        public bool ShareActivity { get; set; }

        protected override bool ProcessOutbound(PresenceSettingEventV2 evt)
        {
            var user = evt.GetUserStatistics();

            if (!user.HasTwitchID)
            {
                LogEvent("Suppressing outbound event. Account has not been merged.");
                return false;
            }

            UserID = user.TwitchID;
            AvailabilityOverride = GetAvailabilityOverride(evt.Availability);
            ShareActivity = evt.ShareActivity;

            FilteredLogger.Log(user.UserID, "Publishing presence settings.");
            return true;
        }

        private string GetAvailabilityOverride(UserConnectionStatus availability)
        {
            switch (availability)
            {
                case UserConnectionStatus.Invisible:
                    return "offline";                                    
                case UserConnectionStatus.Away:
                    return "away";
                case UserConnectionStatus.DoNotDisturb:
                    return "busy";                
                default:
                    return "";
            }
        }

        private UserConnectionStatus GetConnectionStatus(string availability)
        {
            switch(availability)
            {
                case "offline":
                    return UserConnectionStatus.Invisible;
                case "away":
                    return UserConnectionStatus.Away;
                case "busy":
                    return UserConnectionStatus.DoNotDisturb;
                case "":
                    return UserConnectionStatus.Online;
                default:
                    TraceLogger.Warn("Unexpected availability received", availability);
                    return UserConnectionStatus.Online;
            }            
        }


        public override void ProcessInbound()
        {
            var acctProvider = new MergedAccountProvider(UserID);

            string validationError;
            if (!acctProvider.Validate(out validationError))
            {
                LogValidation(validationError);                
                return;
            }

            var user = acctProvider.InternalAccount;
            var userRegion = acctProvider.Region;

            // Share Activity
            try
            {
                var userPrivacy = user.GetPrivacy();
                var newActivityPrivacy = ShareActivity ? ShareActivityPrivacy.Share : ShareActivityPrivacy.Hide;
                if (userPrivacy.ShareActivityPrivacy != newActivityPrivacy)
                {
                    userPrivacy.ShareActivityPrivacy = newActivityPrivacy;
                    userPrivacy.Update(u => u.ShareActivityPrivacy);
                    user.UpdateStatistics();
                    RegionalUserChangeResolver.Create(user.UserID);
                }

                FilteredLogger.Log(user, "Received share activity privacy");
            }
            catch(Exception ex)
            {
                Logger.Warn(ex, "Failed to save user's share activity setting, skipping", new { Contract = this, CurseUserID = acctProvider.ExternalAccount.MergedUserID });
            }


            // Invisible
            var proposedStatus = GetConnectionStatus(AvailabilityOverride);
            if(user.ConnectionStatus != proposedStatus || user.LastConnectionStatus != proposedStatus)
            {
                var ep = new TwitchClientEndpointProvider(user.UserID).Endpoint;
                UserPresenceResolver.CreateAvailabilityOverrideResolver(user.UserID, userRegion.RegionID, ep.MachineKey, proposedStatus);
                FilteredLogger.Log(user, string.Format("Resolving status to {0}", proposedStatus));
            }

            FilteredLogger.Log(acctProvider.InternalAccount, "Processed inbound presence settings event.");
        }

        public override bool Validate()
        {
            if (string.IsNullOrWhiteSpace(UserID))
            {
                LogValidation("Supressing: No UserID");
                return false;
            }
            return true;
        }
    }
}
