﻿using Curse.Friends.Data;
using Curse.Friends.TwitchIdentityMerge;
using System.Linq;
using Curse.Friends.Data.Messaging;
using Curse.Friends.Enums;
using Curse.Logging;
using System;
using Curse.Friends.Tracing;
using System.Collections.Generic;

namespace Curse.Friends.TwitchInterop
{

    internal class WhisperHelper
    {
        private static readonly FilteredUserLogger FilteredLogger = new FilteredUserLogger("WhisperHelper");

        public int UserID { get; set; }
        public int OtherUserID { get; set; }
        public string ConversationID { get; set; }
        public bool IsValid { get; set; }
        public string ValidationMessage { get; set; }
        public object ValidationData { get; set; }

        private WhisperHelper(string validationMessage, object validationData = null)
        {
            IsValid = false;
            ValidationMessage = validationMessage;
            ValidationData = validationData;
        }

        private WhisperHelper(int userID, int otherUserID)
        {            
            UserID = userID;
            OtherUserID = otherUserID;
            ConversationID = PrivateConversation.GenerateConversationID(userID, otherUserID);
            IsValid = true;
        }

        public User GetUser()
        {
            return User.FindByUserID(UserID);
        }

        internal static WhisperHelper GetWhisperInfo(string twitchUserID, string[] participantIDs, bool autoProvisionAccounts)
        {
            if (participantIDs.Any(string.IsNullOrEmpty))
            {
                return new WhisperHelper("Invalid whisper participants.");
            }

            if (participantIDs.Length != 2)
            {
                return new WhisperHelper("Unsupported whisper mode. The ParticipantIDs length is not 2.");
            }

            if (participantIDs.All(p => p == twitchUserID))
            {
                return new WhisperHelper("Invalid whisper participants.");
            }

            var externalAccounts = ExternalAccount.GetDictionaryByTwitchUserIDs(participantIDs);
            if (!externalAccounts.Any(p => p.Value.IsMerged))
            {
                return new WhisperHelper("None of the participants have a merged account.");                
            }

            if (!externalAccounts.Any(p => p.Value.IsMerged && !p.Value.IsAutoProvisioned))
            {
                return new WhisperHelper("All participants are unmerged or auto-provisioned");
            }

            ExternalAccount account;
            if (!autoProvisionAccounts && (!externalAccounts.TryGetValue(twitchUserID, out account) || !account.IsMerged))
            {
                return new WhisperHelper("The requesting user does not have a merged account.");
            }
            
            var userID = 0;
            var otherUserID = 0;

            // Ensure all participants have a mapped account
            foreach (var id in participantIDs)
            {
                int curseUserID;


                if (!externalAccounts.TryGetValue(id, out account) || !account.IsMerged)
                {
                    if (!autoProvisionAccounts)
                    {
                        return new WhisperHelper("Not auto-provisioning an account because autoProvisionAccounts is disabled");
                    }

                    FilteredLogger.Log(twitchUserID, "Auto-provisioning a participant account", new { Sender = twitchUserID, UnmergedParticipant = id });

                    var mergeState = IdentityMergeAuthHelper.ProvisionOrGetMappedUser(id, true);

                    if (mergeState.Status != IdentityMergeStatus.Merged && mergeState.Status != IdentityMergeStatus.AutoProvisioned)
                    {
                        return new WhisperHelper("Failed to provision an account", mergeState); 
                    }

                    curseUserID = mergeState.CurseUserID;
                }
                else
                {
                    curseUserID = account.MergedUserID;
                }

                if (id == twitchUserID)
                {
                    userID = curseUserID;
                }
                else
                {
                    otherUserID = curseUserID;
                }
            }

            if (userID == 0 || otherUserID == 0)
            {
                return new WhisperHelper("Failed to resolve Curse users from Twitch users.");                                 
            }

            if (userID == otherUserID)
            {
                return new WhisperHelper("User IDs are the same, and should be a distinct pair.");
            }

            return new WhisperHelper(userID, otherUserID);
        }

        public void ToggleMuted(bool isMuted)
        {
            var conversation = ConversationManager.GetConversationContainer(UserID, ConversationID);
            if (conversation == null)
            {                
                return;
            }

            var conversationParent = conversation.GetConversationParent(UserID);
            conversationParent.ToggleMuted(isMuted);
        }

        public void ToggleHidden(bool isHidden)
        {
            var conversation = ConversationManager.GetConversationContainer(UserID, ConversationID);
            if (conversation == null || !conversation.CanHide())
            {
                return;
            }

            var conversationParent = conversation.GetConversationParent(UserID);
            conversationParent.ToggleHidden(isHidden);
        }

        public ConversationMessageEmoteSubstitution[] GetEmoteSubstitutionOverrides(EmoticonSubstitutionContract[] emoticons, string messageBody)
        {
            if (emoticons != null && emoticons.Length > 0)
            {
                var ids = new HashSet<long>(emoticons.Select(e => e.ID));
                var emotes = TwitchEmote.MultiGetByIDs(ids).ToDictionary(e => e.EmoteID);

                var overrides = new List<ConversationMessageEmoteSubstitution>();
                var resolvedEmotes = new List<ConversationMessageEmoteSubstitution>();
                foreach (var substitution in emoticons.Where(s => s.Start >= 0 && s.End < messageBody?.Length))
                {
                    var resolved = new ConversationMessageEmoteSubstitution
                    {
                        EmoteID = substitution.ID,
                        StartIndex = substitution.Start,
                        EndIndex = substitution.End,
                        MessageText = messageBody.Substring(substitution.Start, substitution.End - substitution.Start + 1)
                    };

                    TwitchEmote emote;
                    if (emotes.TryGetValue(substitution.ID, out emote))
                    {
                        resolved.EmoteHeight = emote.Height;
                        resolved.EmoteWidth = emote.Width;
                        resolved.EmoteSet = emote.EmoticonSet;
                    }
                    resolvedEmotes.Add(resolved);
                }
                return resolvedEmotes.ToArray();
            }
            return null;
        }
    }
}
