﻿using System;
using Amazon.Kinesis.Model;
using Curse.Extensions;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.TwitchInteropService.Stats;
using Curse.Logging;

namespace Curse.Friends.TwitchInteropService.Identity
{
    public class IdentityStreamProcessor
    {
       
        public static void Process(IdentityEvent identityEvent)
        {
            switch (identityEvent.EventType)
            {
                case "user_password_changed":
                case "user_token_revoked":
                case "user_authorization_deleted":
                    InvalidateToken(identityEvent.UserID, identityEvent.CanonicalClientID, identityEvent.Timestamp.FromEpochSeconds());
                    break;
                default:
                    Logger.Warn("Unknown Identity Event Type", new {identityEvent});
                    break;
            }
        }

        private static void InvalidateToken(string userID, string clientID, DateTime timestamp)
        {
            var account = ExternalAccount.GetLocal(userID, AccountType.Twitch);

            if (account == null)
            {
                Logger.Debug("Skipping token invalidation event. There is no local account for this Twitch ID.", new { userID });
                StatsTracker.IdentityStreamStats.EventSkipped();
                return;
            }

            var accountHasID = !string.IsNullOrEmpty(account.ClientID);
            if (!string.IsNullOrEmpty(clientID) && ((accountHasID && clientID != account.ClientID) || (!accountHasID && clientID != FriendsServiceConfiguration.Instance.TwitchClientID)))
            {
                StatsTracker.IdentityStreamStats.EventSkipped();
                Logger.Debug("Skipping token invalidation event. It is for an unrelated client ID.", new { SuppliedClientID = clientID, OurClientID = FriendsServiceConfiguration.Instance.TwitchClientID });
                return;
            }

            account.MarkForReauthentication();                

            if (account.MergedUserID > 0)
            {
                var userStats = UserStatistics.GetByUserID(account.MergedUserID);
                if (userStats == null)
                {
                    return;
                }

                var timestampMs = timestamp.ToEpochMilliseconds();
                // Only move the timestamp forward
                if (userStats.TokenTimestamp < timestampMs)
                {
                    userStats.TokenTimestamp = timestampMs;
                    userStats.Update(p => p.TokenTimestamp);
                }

            }

            StatsTracker.IdentityStreamStats.EventProcessed();                 
            
        }
    }
}
