﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.CloudServices.Jobs;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.Models;
using Curse.Friends.TwitchApi;
using Curse.Logging;

namespace Curse.Friends.TwitchInteropService.Jobs
{
    class VerifiedBotSyncJob : BaseJob
    {
        public override JobScheduleMode ScheduleMode
        {
            get { return JobScheduleMode.Interval; }
        }

        public override int RunFrequencyMinutes
        {
            get { return 10; }
        }

        public override bool OnlyDefaultRegion
        {
            get { return true; }
        }

        private static readonly LogCategory Logger = new LogCategory("VerifiedBotSyncJob");

        public override void Run()
        {
            // Hit up the Curse shim
            using (var client = new CurseShimClient(FriendsServiceConfiguration.Instance.TwitchShimApiKey, FriendsServiceConfiguration.Instance.TwitchShimUrl))
            {
                var resp = client.VerifiedBots();

                if (resp.Status != TwitchResponseStatus.Success)
                {
                    Logger.Error("Failed to get the verified bot list from Twitch", resp);
                    return;
                }

                // Get all mapped / external accounts
                var externalAccounts = ExternalAccount.GetByTwitchUserIDs(resp.Value.VerifiedBotIDs);

                var verifiedBotUsers = new HashSet<int>(externalAccounts.Select(p => p.MergedUserID));

                // Get all bots
                var allBots = Bot.GetAllLocal(p => p.IsVerified, true);

                // Unverify any bots not in the list
                var unverifiedBots = allBots.Where(p => !verifiedBotUsers.Contains(p.UserID)).ToArray();
                                   
                foreach (var bot in unverifiedBots)
                {
                    Logger.Info("Unverifying existing bot: " + bot.UserID);
                    bot.IsVerified = false;
                    bot.DateModified = DateTime.UtcNow;
                    bot.Update();
                }
                

                // Update all mapped accounts
                foreach (var userID in verifiedBotUsers)
                {
                    var bot = Bot.GetByUserID(userID);
                    if (bot == null)
                    {
                        bot = new Bot
                        {
                            DateCreated = DateTime.UtcNow,
                            UserID = userID,
                            IsVerified = true
                        };
                        bot.InsertLocal();
                        Logger.Info("Inserting new verified bot: " + userID);
                    }
                    else if(!bot.IsVerified)
                    {
                        bot.IsVerified = true;
                        bot.DateModified = DateTime.UtcNow;
                        bot.Update();
                        Logger.Info("Marking existing bot as verified: " + userID);
                    }
                }

            }
            
        }
    }
}
