﻿using System;
using System.Threading;
using Amazon;
using Amazon.Kinesis;
using Amazon.Runtime;
using Amazon.SecurityToken;
using Amazon.SecurityToken.Model;
using Curse.Friends.TwitchInteropService.Configuration;

namespace Curse.Friends.TwitchInteropService.Kinesis
{
    public class KinesisClient : IDisposable
    {
        private readonly AmazonKinesisClient _client;
        private readonly IDisposable[] _disposables;

        private KinesisClient(AmazonKinesisClient client, params IDisposable[] disposables)
        {
            _client = client;
            _disposables = disposables;
        }

        public AmazonKinesisClient Client { get { return _client; } }

        public static KinesisClient CreateBasicClient(string accessID, string secret, string regionEndpointName)
        {
            var credentials = new BasicAWSCredentials(accessID, secret);
            var client = new AmazonKinesisClient(credentials, RegionEndpoint.GetBySystemName(regionEndpointName));
            return new KinesisClient(client);
        }

        public static KinesisClient CreateAssumeRoleClient(string streamName, string accessID, string secret, string roleArn, string regionEndpointName)
        {
            var stsClient = new AmazonSecurityTokenServiceClient(new BasicAWSCredentials(accessID, secret), RegionEndpoint.GetBySystemName(regionEndpointName));
            var assumeRoleRequest = new AssumeRoleRequest { RoleArn = roleArn, RoleSessionName = string.Format("{0}_{1}_consumer", Environment.MachineName, streamName) };
            var credentials = new STSAssumeRoleAWSCredentials(stsClient, assumeRoleRequest);
            var client = new AmazonKinesisClient(credentials, RegionEndpoint.GetBySystemName(regionEndpointName));
            return new KinesisClient(client, credentials, stsClient);
        }

        public static KinesisClient CreateInstanceProfileClient(string streamName, string roleArn, string regionEndpointName)
        {
            var stsClient = new AmazonSecurityTokenServiceClient(new InstanceProfileAWSCredentials());
            var assumeRoleRequest = new AssumeRoleRequest{RoleArn = roleArn, RoleSessionName = string.Format("{0}_{1}_consumer", Environment.MachineName, streamName)};
            var credentials = new STSAssumeRoleAWSCredentials(stsClient, assumeRoleRequest);
            var client = new AmazonKinesisClient(credentials, RegionEndpoint.GetBySystemName(regionEndpointName));
            return new KinesisClient(client);
        }

        private bool _isDisposed;
        private readonly object _disposeLock = new object();

        public void Dispose()
        {
            var acquired = Monitor.TryEnter(_disposeLock, TimeSpan.FromSeconds(5));

            if (!acquired)
            {
                return;
            }

            if (_isDisposed)
            {
                return;
            }

            try
            {
                _isDisposed = true;

                _client.Dispose();
                foreach (var disposable in _disposables)
                {
                    disposable.Dispose();
                }

            }
            finally
            {
                Monitor.Exit(_disposeLock);
            }
        }


        public static KinesisClient CreateClient(TwitchInteropConnectionInfo kinesisInfo)
        {
            switch (kinesisInfo.CredentialType)
            {
                case AwsCredentialType.AssumeRole:
                    return CreateAssumeRoleClient(kinesisInfo.StreamName, kinesisInfo.UserAccessKey, kinesisInfo.UserAccessSecret, kinesisInfo.RoleArn, kinesisInfo.RegionEndpoint);                    
                case AwsCredentialType.InstanceProfile:
                    return CreateInstanceProfileClient(kinesisInfo.StreamName, kinesisInfo.RoleArn, kinesisInfo.RegionEndpoint);                    
                default:
                    return CreateBasicClient(kinesisInfo.UserAccessKey, kinesisInfo.UserAccessSecret, kinesisInfo.RegionEndpoint);                    
            }            
        }       
    }
}
