﻿using System;
using System.Linq;
using System.Net;
using System.Threading;
using Amazon.Kinesis;
using Amazon.Kinesis.Model;
using Curse.Friends.Data;
using Curse.Friends.TwitchInteropService.Configuration;
using Curse.Logging;

namespace Curse.Friends.TwitchInteropService.Kinesis
{
    class KinesisConsumer
    {
        private readonly CancellationTokenSource _cancellationTokenSource;
        private readonly KinesisClient _client;
        private readonly Thread _thread;
        private readonly KinesisShardIterator _shardIterator;
        private readonly Action<Record> _callback;
        private readonly LogCategory Logger;

        public KinesisConsumer(TwitchInteropConnectionInfo config, KinesisShardIterator shardIterator, Action<Record> callback)
        {
            Logger = new LogCategory(string.Format("{0}-{1}", shardIterator.StreamName, shardIterator.ShardID)) { Throttle = TimeSpan.FromSeconds(30) };

            _cancellationTokenSource = new CancellationTokenSource();
            _shardIterator = shardIterator;
            _client = KinesisClient.CreateClient(config);
            _callback = callback;                        
            _thread = new Thread(ConsumeKinesisStream);
        }

        public void Start()
        {
            _thread.Start();
        }

        public void Stop()
        {
            _cancellationTokenSource.Cancel();

            if (_thread.IsAlive)
            {
                _thread.Join(TimeSpan.FromSeconds(5));
            }
        }

        private void ConsumeKinesisStream()
        {            

            // Describe the stream
            var resp = _client.Client.DescribeStream(new DescribeStreamRequest
            {
                StreamName = _shardIterator.StreamName
            });

            if (resp.HttpStatusCode != HttpStatusCode.OK)
            {
                throw new InvalidOperationException("Failed to get Kinesis Stream details for " + _shardIterator.StreamName);
            }
            
            var lastProcessedSequenceNumber = _shardIterator.LastSequenceNumber;
            var iterator = GetInitialIterator(lastProcessedSequenceNumber);

            while (!_cancellationTokenSource.Token.IsCancellationRequested)
            {
                if (iterator == null)
                {
                    Logger.Warn("Shard Iterator is null, shard has been closed!");
                    break;
                }

                try
                {
                    var delay = TimeSpan.FromSeconds(1);

                    var getRecordsRequest = new GetRecordsRequest
                    {
                        ShardIterator = iterator,
                        Limit = 10000
                    };

                    var getRecordsResponse = _client.Client.GetRecords(getRecordsRequest);
                    if (getRecordsResponse.HttpStatusCode != HttpStatusCode.OK)
                    {
                        Logger.Warn("Error retrieving records from Kinesis",
                            new
                            {
                                getRecordsResponse.HttpStatusCode,
                                getRecordsResponse.ResponseMetadata
                            });

                        _cancellationTokenSource.Token.WaitHandle.WaitOne(TimeSpan.FromSeconds(30));
                        continue;
                    }

                    // If we're more than 10 seconds behind, log it
                    if (getRecordsResponse.MillisBehindLatest > 10000)
                    {
                        var tsBehind = TimeSpan.FromMilliseconds(getRecordsResponse.MillisBehindLatest);
                        // Lower the delay to try to catch up
                        delay = TimeSpan.FromMilliseconds(1);
                        Logger.Warn("Kinesis stream is " + tsBehind.TotalSeconds.ToString("N0") + " seconds behind latest");
                    }

                    var recordsToProcess = getRecordsResponse.Records.Where(r => string.CompareOrdinal(r.SequenceNumber, lastProcessedSequenceNumber) > 0).ToArray();


                    var itemsProcessed = false;
                    try
                    {
                        foreach (var record in recordsToProcess)
                        {
                            _callback(record);
                            lastProcessedSequenceNumber = record.SequenceNumber;
                            itemsProcessed = true;
                        }
                        iterator = getRecordsResponse.NextShardIterator;
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, "Error processing Kinesis callback");
                    }

                    if (itemsProcessed)
                    {
                        _shardIterator.LastSequenceNumber = lastProcessedSequenceNumber;
                        _shardIterator.Update(s => s.LastSequenceNumber);
                    }

                    _cancellationTokenSource.Token.WaitHandle.WaitOne(delay);
                }
                catch (ExpiredIteratorException ex)
                {
                    Logger.Warn(ex, "The kinesis stream iterator has expired. Recreating it...");

                    try
                    {
                        iterator = GetInitialIterator(lastProcessedSequenceNumber);
                    }
                    catch (Exception ex2)
                    {
                        Logger.Error(ex2, "Failed to re-create kinesis iterator, after it had expired. Waiting 10 seconds before retrying...");
                        _cancellationTokenSource.Token.WaitHandle.WaitOne(TimeSpan.FromSeconds(10));
                    }

                }
                catch (OperationCanceledException)
                {
                    return;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Error getting records from Kinesis");
                }
            }
        }

        private string GetInitialIterator(string lastProcessedSequenceNumber)
        {
            var getIteratorRequest = new GetShardIteratorRequest
            {
                StreamName = _shardIterator.StreamName,
                ShardId = _shardIterator.ShardID,
                ShardIteratorType = ShardIteratorType.TRIM_HORIZON
            };

            if (!string.IsNullOrEmpty(lastProcessedSequenceNumber))
            {
                getIteratorRequest.ShardIteratorType = ShardIteratorType.AFTER_SEQUENCE_NUMBER;
                getIteratorRequest.StartingSequenceNumber = lastProcessedSequenceNumber;
            }

            var attempt = 0;
            while (true)
            {
                var getIteratorResponse = _client.Client.GetShardIterator(getIteratorRequest);

                if (getIteratorResponse.HttpStatusCode != HttpStatusCode.OK)
                {
                    if (attempt > 2 || attempt++ > 2)
                    {
                        // Default to starting over completely
                        getIteratorRequest.ShardIteratorType = ShardIteratorType.TRIM_HORIZON;
                        getIteratorRequest.StartingSequenceNumber = null;
                    }
                    try
                    {
                        _cancellationTokenSource.Token.WaitHandle.WaitOne(TimeSpan.FromSeconds(10));
                    }
                    catch (OperationCanceledException)
                    {
                        return null;
                    }
                }
                else
                {
                    return getIteratorResponse.ShardIterator;
                }
            }
        }
    }
}
