﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Amazon.Kinesis.Model;
using Curse.Friends.TwitchInteropService.Configuration;
using Curse.Friends.TwitchInteropService.Identity;
using Newtonsoft.Json;
using Curse.Friends.TwitchInteropService.UserMutation;
using Curse.Logging;
using Curse.Friends.Data;
using System.Net;

namespace Curse.Friends.TwitchInteropService.Kinesis
{
    class KinesisManager
    {        
        private static readonly Dictionary<string, TwitchInteropConnectionInfo> _configs;
        private static readonly Dictionary<string, Action<Record>> _callbacks = new Dictionary<string, Action<Record>>        
        {
            { "session-invalidations-stream", ProcessIdentityRecord },
            { "user-mutations-stream", ProcessUserMutationRecord }
        };

        private static readonly ConcurrentDictionary<string, KinesisConsumer> _consumers = new ConcurrentDictionary<string, KinesisConsumer>();
        
        static KinesisManager()
        {
            _configs = TwitchInteropConfiguration.Instance.KinsesisStreams.ToDictionary(p => p.StreamName);
        }

        public static void Start()
        {
            // Get the shards for all configured Kinesis streams
            RefreshKinesisShards();
        }

        public static void UnhostShard(KinesisShardIterator iterator)
        {
            try
            {
                KinesisConsumer consumer;
                var iteratorKey = iterator.GetGeneratedKey();
                if (!_consumers.TryGetValue(iteratorKey, out consumer))
                {
                    return;
                }

                Logger.Info("Unhosting shard: " + iterator.DisplayName);
                consumer.Stop();
                _consumers.TryRemove(iteratorKey, out consumer);
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
           
        }

        public static void HostShard(KinesisShardIterator iterator)
        {
            try
            {
                var iteratorKey = iterator.GetGeneratedKey();

                if (_consumers.ContainsKey(iteratorKey))
                {                    
                    return;
                }

                Logger.Info("Hosting shard: " + iterator.DisplayName);
                var config = _configs[iterator.StreamName];
                var callback = _callbacks[iterator.StreamName];
                var consumer = new KinesisConsumer(config, iterator, callback);

                if (_consumers.TryAdd(iteratorKey, consumer))
                {
                    consumer.Start();
                }


            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
                      
        }

        public static void Stop()
        {            
            foreach (var consumer in _consumers)
            {
                consumer.Value.Stop();                
            }            
        }

        public static void RefreshKinesisShards()
        {
            foreach (var config in _configs.Values)
            {
                try
                {
                    RefreshKinesisShard(config);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to refresh Kinesis shards for supplied config", new { config.StreamName });
                }

            }

        }

        private static void RefreshKinesisShard(TwitchInteropConnectionInfo config)
        {
            using (var client = KinesisClient.CreateClient(config))
            {
                var resp = client.Client.DescribeStream(new DescribeStreamRequest
                {
                    StreamName = config.StreamName
                });

                if (resp.HttpStatusCode != HttpStatusCode.OK)
                {
                    throw new InvalidOperationException("Failed to get Kinesis Stream details for " + config.StreamName);
                }

                var streamDescription = resp.StreamDescription;

                foreach (var shard in streamDescription.Shards)
                {
                    var shardIteror = KinesisShardIterator.GetBySourceNameAndShard(config.Account, streamDescription.StreamName, shard.ShardId);

                    if (shardIteror == null)
                    {
                        shardIteror = new KinesisShardIterator
                        {
                            AccountSource = config.Account,
                            StreamName = streamDescription.StreamName,
                            ShardID = shard.ShardId,
                            LastSequenceNumber = null,
                            RegionID = KinesisShardIterator.LocalConfigID
                        };

                        shardIteror.InsertLocal();
                    }
                    else
                    {
                        shardIteror.RegionID = KinesisShardIterator.LocalConfigID;
                        shardIteror.Update(p => p.RegionID);
                    }
                }
            }
        }

        #region Identity

        private static void ProcessIdentityRecord(Record record)
        {
            string dataString= null;
            IdentityEvent identityEvent = null;
            try
            {
                dataString = Encoding.UTF8.GetString(record.Data.ToArray());
                identityEvent = JsonConvert.DeserializeObject<IdentityEvent>(dataString);
                IdentityStreamProcessor.Process(identityEvent);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process identity record.", new { dataString, identityEvent });
            }
           
        }

        private static void ProcessUserMutationRecord(Record record)
        {
            string dataString = null;
            UserMutationEvent identityEvent = null;
            try
            {
                dataString = Encoding.UTF8.GetString(record.Data.ToArray());
                identityEvent = JsonConvert.DeserializeObject<UserMutationEvent>(dataString);
                UserMutationProcessor.Process(identityEvent);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process identity record.", new { dataString, identityEvent });
            } 
        }

        #endregion
    }
}

