﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Data.Queues;
using Curse.Friends.Enums;
using Curse.Friends.Tracing;
using Curse.Friends.TwitchApi;

namespace Curse.Friends.TwitchInteropService.Processors
{
    class TwitchCommunityModsProcessor
    {
        private static readonly FilteredUserLogger Logger = new FilteredUserLogger("ModsProcessor");
        private static readonly CurseShimClient shimClient = new CurseShimClient(FriendsServiceConfiguration.Instance.TwitchShimApiKey, FriendsServiceConfiguration.Instance.TwitchShimUrl);

        public static void Process(TwitchCommunityModsWorker worker)
        {
            Logger.Log(worker.TwitchID, "Running Moderators sync", worker);

            if (string.IsNullOrWhiteSpace(worker.TwitchID))
            {
                Logger.Warn(null, null, null, "TwitchID not provided", worker);
            }

            var account = ExternalAccount.GetByTwitchUserID(worker.TwitchID);
            if (account == null)
            {
                Logger.Warn(null, null, null, "Unable to sync mods, account not found", worker);
                return;
            }

            var community = ExternalCommunity.GetByTwitchID(worker.TwitchID);
            if (community == null)
            {
                Logger.Warn(null, null, null, "Unable to sync mods, community not found", worker);
                return;
            }

            if (!worker.ForceSync && !community.ShouldSyncMods())
            {
                Logger.Log(null, account, "Skipping mod sync, it has been done too recently", worker);
                return;
            }

            SyncMods(account, community);

        }

        private static void SyncMods(ExternalAccount account, ExternalCommunity community)
        {
            try
            {
                var modsResponse = shimClient.GetChannelMods(account.ExternalID);
                if (modsResponse.Status != TwitchResponseStatus.Success)
                {
                    switch (modsResponse.Status)
                    {
                        case TwitchResponseStatus.NotFound:
                        case TwitchResponseStatus.Unprocessable:
                        case TwitchResponseStatus.TwitchServerError:
                            Logger.Log(null, account, "Unable to get mod list", new { account.ExternalID, account.ExternalUsername, modsResponse });
                            break;
                        default:
                            Logger.Warn(null, account, "Unable to get mod list due to unknown response status: " + modsResponse.Status, new { account.ExternalID, account.ExternalUsername, modsResponse });
                            break;
                    }
                    return;
                }

                if (modsResponse.Value == null || modsResponse.Value.ModIDs == null)
                {
                    Logger.Warn(null, account, "Unable to sync mods, response is invalid", new { Account = account.GetLogData(), modsResponse });
                    return;
                }

                UpdateModerators(account, community, new HashSet<string>(modsResponse.Value.ModIDs));
            }
            finally
            {
                ExternalCommunityMemberSyncWorker.Create(community.RegionID, community.ExternalID, community.Type, GroupRoleTag.SyncedModerator);
            }
        }

        private static void UpdateModerators(ExternalAccount account, ExternalCommunity community, HashSet<string> moderatorIDs)
        {
            var now = DateTime.UtcNow;

            var existingMemberships = ExternalCommunityMembership.GetAllLocal(m => m.CommunityTypeAndRoleIndex, community.ExternalID + AccountType.Twitch + GroupRoleTag.SyncedModerator)
                .ToDictionary(m => m.ExternalUserID);

            // Don't trust an empty mod list if we think there are currently 4 or more active mods
            if (moderatorIDs.Count == 0 && existingMemberships.Count(m=>m.Value.Status != ExternalCommunityMembershipStatus.Deleted) >= 4)
            {
                Logger.Log(null, account, "Skipping mod sync, empty mod list and 4 or more were active as of the last sync - not trusting results", new { community = community.GetLogData() });
                return;
            }

            var mods = ExternalCommunity.MultiGetLocal(moderatorIDs.Select(id => new KeyInfo(id, AccountType.Twitch))).ToDictionary(c => c.ExternalID, c => new NewCommunityMember
            {
                RoleDate = now,
                ExternalUserID = c.ExternalID,
                ExternalUsername = c.ExternalName,
                ExternalUserDisplayName = c.ExternalDisplayName
            });

            community.ResolveMemberships(existingMemberships, mods, GroupRoleTag.SyncedModerator);
            community.ModSyncTimestamp = now.ToEpochMilliseconds();
            community.Update(s => s.ModSyncTimestamp);

            Logger.Log(null, account, "Moderators synced", new {Community = community.GetLogData(), ModCount = moderatorIDs.Count});
        }
    }
}
