﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Friends.Enums;
using Curse.Friends.Tracing;
using Curse.Friends.TwitchInteropService.Sqs;
using Curse.Friends.UserEvents;
using Curse.Logging;
using Newtonsoft.Json;
using Curse.Friends.TwitchInterop;
using Curse.Friends.TwitchInteropService.Configuration;
using Microsoft.CSharp.RuntimeBinder;
using Curse.Friends.TwitchInteropService.Stats;

namespace Curse.Friends.TwitchInteropService.Processors
{
    class TwitchInteropEventWorkerProcessor
    {
        private static readonly LogCategory Logger = new LogCategory("UserEventWorker");
        private static readonly LogCategory TraceLogger = new LogCategory("Outbound") { AlphaLevel = LogLevel.Trace, Throttle = TimeSpan.FromSeconds(30) };
        private static readonly LogCategory DebugLogger = new LogCategory("Outbound") { Throttle = TimeSpan.FromMinutes(1), ReleaseLevel = LogLevel.Debug };
        private static readonly Dictionary<string, UserEventTypeMap> _eventTypeMap = new Dictionary<string, UserEventTypeMap>();
        private static readonly FilteredUserLogger FilteredLogger = new FilteredUserLogger("Outbound");

        public static void Initialize()
        {
            try
            {
                // Get all interop event types
                var contractTypes = typeof(BaseTwitchInteropContract).Assembly.GetTypes()
                                    .Where(p => typeof(BaseTwitchInteropContract).IsAssignableFrom(p) && !p.IsAbstract)
                                    .ToArray();

                foreach (var type in contractTypes)
                {
                    var genericArgs = type.BaseType.GetGenericArguments();
                    var typeContract = genericArgs[0];
                    var typeEvent = genericArgs[1];

                    _eventTypeMap.Add(typeEvent.Name, new UserEventTypeMap(typeContract, typeEvent));
                }
                
            }
            catch (Exception ex)
            {
                Logger.Fatal(ex, "Failed to initialize!");
            }

        }

        private static void LogEvent(LogLevel level, string message, BaseUserEvent e = null, Exception ex = null)
        {
            DebugLogger.Log(level, DateTime.UtcNow, ex, message, e?.GetLogData());
        }

        public static void Process(UserEventWorker e)
        {
            StatsTracker.UserEventStats.TrackEvent(e.EventType);

            UserEventTypeMap eventType;
            if (!_eventTypeMap.TryGetValue(e.EventType, out eventType))
            {
                LogEvent(LogLevel.Warn, "Unknown event type: " + e.EventType);
                return;
            }

            // Deserialize the event
            var evt = (BaseUserEvent)JsonConvert.DeserializeObject(e.EventData, eventType.EventType);

            var pairEvt = evt as BaseUserPairEvent;
            var convEvt = evt as BaseConversationUserEvent;
            if (pairEvt != null)
            {
                var pair = pairEvt.GetUserPair();
                if (!pair.BothParticipantsMerged)
                {
                    LogEvent(LogLevel.Trace, "Skipping event because at least one user is not merged");
                    return;
                }
            }
            else if (convEvt != null)
            {
                var pair = convEvt.GetSenderAndRecipient();
                if (!pair.BothParticipantsMerged)
                {
                    LogEvent(LogLevel.Trace, "Skipping event because at least one user is not merged");
                    return;
                }
            }
            else
            {
                if (!evt.IsUserMerged())
                {
                    LogEvent(LogLevel.Trace, "Skipping event because user is not merged: " + e.EventType, evt);
                    return;
                }
            }

            // Create the contract
            var contract = (BaseTwitchInteropContract)Activator.CreateInstance(eventType.ContractType);
            try
            {
                if (!contract.ProcessOutbound(evt))
                {
                    LogEvent(LogLevel.Trace, "Skipping event: " + e.EventType, evt);
                    return;
                }
            }
            catch (Exception ex)
            {
                LogEvent(LogLevel.Error, "Failed to process user event due to an unhandled exception.", evt, ex);
                return;
            }
            
            var serializedContract = contract.Serialize();
            var contractType = contract.GetContractName();
            
            if (TwitchInteropConfiguration.Instance.PublisherEnabled)
            {                                
                var publisher = SqsManager.GetPublisher();
                publisher.SendEvent(contractType, serializedContract);
                contract.LogEvent("Publishing event...");
            }
            else
            {
                LogEvent(LogLevel.Trace, "Publisher disabled. Skipping event: " + e.EventType, evt);
            }            
        }

        private class UserEventTypeMap
        {

            public UserEventTypeMap(Type contractType, Type eventType)
            {
                ContractType = contractType;
                EventType = eventType;
            }

            public Type ContractType { get; set; }
            public Type EventType { get; set; }

        }

    }


}
