﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Friends.TwitchApi;
using Curse.Friends.TwitchApi.CurseShim;
using Curse.Friends.Data;
using Curse.Friends.Data.Models;
using Curse.Aerospike;
using Curse.Friends.Configuration;
using Curse.Logging;
using Curse.Friends.Enums;
using Curse.Friends.Tracing;
using Curse.Friends.TwitchInterop;
using Curse.Friends.TwitchInteropService.Configuration;
using User = Curse.Friends.Data.User;

namespace Curse.Friends.TwitchInteropService.Processors
{
    public class TwitchMergeWorkerProcessor
    {
        private static readonly CurseShimClient _shimClient = new CurseShimClient(FriendsServiceConfiguration.Instance.TwitchShimApiKey, FriendsServiceConfiguration.Instance.TwitchShimUrl);
        private static readonly FilteredUserLogger FilteredLogger = new FilteredUserLogger("TwitchMergeWorkerProcessor");
        
        public static void Process(TwitchMergeWorker worker)
        {
            var userRegion = UserRegion.GetLocal(worker.UserID);
            if (userRegion == null)
            {
                Logger.Warn("No user region for Merge worker", worker);
                return;
            }

            var user = userRegion.GetUser();
            if (user == null)
            {
                Logger.Warn("No user for Merge worker", worker);
                return;
            }

            if (!user.IsMerged || string.IsNullOrEmpty(user.TwitchID))
            {
                Logger.Error("Attempt to merge details for an unmerged user", new { Worker = worker, IsMerged = user.IsMerged, TwitchID = user.TwitchID, IsProvisional = user.IsProvisional, IsTempAccount = user.IsTempAccount, HasSyncedAccount = user.HasSyncedAccount });
                return;
            }

            if (TwitchInteropConfiguration.Instance.UseWhitelist && !TracedUsers.IsWhitelisted(user.TwitchID))
            {
                // Reset this, so user's don't get stuck in this state
                user.LastSocialSync = DateTime.MinValue;
                user.Update(p => p.LastSocialSync);
                
                FilteredLogger.Log(user, null, "Skipping merge_account. User is not whitelisted");
                return;
            }

            if (!user.ShouldSyncSocially())
            {
                FilteredLogger.Log(user, null, "Skipping merge_account. User has synced recently.");
                return;
            }

            user.LastSocialSync = DateTime.UtcNow;
            user.Update(p => p.LastSocialSync);

            var privacy = UserPrivacySettings.GetByUserOrDefault(user.UserID);

            var request = CreateRequest(user, userRegion, privacy);
            FilteredLogger.Log(user, null, "Sending merge_account payload", new { user.UserID, user.Username, request });

            var response = _shimClient.MergeAccount(request);
            ProcessResponse(response, user, userRegion, privacy);
        }

        private static MergeAccountPayload CreateRequest(User user, UserRegion userRegion, UserPrivacySettings privacy)
        {
            var blocks = UserBlock.GetAllLocal(b => b.UserID, user.UserID).Where(b=>b.Status == UserBlockStatus.Active);
            var friends = Friendship.GetAll(userRegion.RegionID, f => f.UserID, user.UserID);

            var blockedIDs = new HashSet<int>(blocks.Select(b => b.OtherUserID));
            var friendIDs = new HashSet<int>(friends.Where(f => f.Status == FriendshipStatus.Confirmed).Select(f => f.OtherUserID));
            var requestIDs = new HashSet<int>(friends.Where(f => f.Status == FriendshipStatus.AwaitingThem).Select(f => f.OtherUserID));
            var allIDs = new HashSet<int>(blockedIDs.Concat(friendIDs).Concat(requestIDs));

            var allUserRegions = UserRegion.MultiGetLocal(allIDs.Select(id => new KeyInfo(id))).ToDictionary(r => r.UserID);
            var allUsers = allUserRegions.Values.GroupBy(r => r.RegionID).SelectMany(g => User.MultiGet(g.Key, g.Select(r => new KeyInfo(r.UserID))))
                .Where(u => u.IsMerged).ToDictionary(u => u.UserID);

            var blockUsers = new HashSet<string>(allUsers.Where(u => blockedIDs.Contains(u.Key)).Select(u => u.Value.TwitchID));
            var friendUsers = new HashSet<string>(allUsers.Where(u => friendIDs.Contains(u.Key)).Select(u => u.Value.TwitchID));
            var requestedUsers = new HashSet<string>(allUsers.Where(u => requestIDs.Contains(u.Key)).Select(u => u.Value.TwitchID));

            // Filter out non-whitelisted people
            if (TwitchInteropConfiguration.Instance.UseWhitelist)
            {
                FilteredLogger.Log(user, null, "Filtering out non-whitelisted users from request");
                blockUsers = new HashSet<string>(blockUsers.Intersect(TracedUsers.WhitelistedUserIDs));
                friendUsers = new HashSet<string>(friendUsers.Intersect(TracedUsers.WhitelistedUserIDs));
                requestedUsers = new HashSet<string>(requestedUsers.Intersect(TracedUsers.WhitelistedUserIDs));
            }

            return new MergeAccountPayload
            {
                UserID = user.TwitchID,
                BlockIDs = blockUsers.ToArray(),
                FriendIDs = friendUsers.ToArray(),
                FriendRequestIDs = requestedUsers.ToArray(),
                AccountSettings = new MergeAccountSettings
                {
                    ShareActivity = privacy.ShareActivityPrivacy == ShareActivityPrivacy.Share,
                    IgnoreWhispersFromStrangers = privacy.BlockStrangerPMs,
                    IsInvisible = user.ConnectionStatus == UserConnectionStatus.Invisible,
                    IsWhisperBanned = UserBan.IsBannedFrom(user.UserID, UserBanType.Whisper)
                }
            };
        }

        private static void ProcessResponse(TwitchResponse<MergeAccountPayload> response, User user, UserRegion userRegion, UserPrivacySettings privacy)
        {
            if (response.Status != TwitchResponseStatus.Success)
            {
                FilteredLogger.Warn(user, null, $"Merge request failed for {user.Username} ({user.UserID})", response);
                return;
            }

            FilteredLogger.Log(user, null, "Processing merge_account response", new { response = response.Value });
            
            var blockedIDs = new HashSet<string>(response.Value.BlockIDs ?? new string[0]);
            var friendIDs = new HashSet<string>(response.Value.FriendIDs ?? new string[0]);
            var requestIDs = new HashSet<string>(response.Value.FriendRequestIDs ?? new string[0]);

            // Filter out non-whitelisted people
            if (TwitchInteropConfiguration.Instance.UseWhitelist)
            {
                FilteredLogger.Log(user, null, "Filtering non-whitelisted users from response");
                blockedIDs = new HashSet<string>(blockedIDs.Intersect(TracedUsers.WhitelistedUserIDs));
                friendIDs = new HashSet<string>(friendIDs.Intersect(TracedUsers.WhitelistedUserIDs));
                requestIDs = new HashSet<string>(requestIDs.Intersect(TracedUsers.WhitelistedUserIDs));
            }

            var allAccounts = ExternalAccount.MultiGetLocal(new HashSet<string>(blockedIDs.Concat(friendIDs).Concat(requestIDs)).Select(id => new KeyInfo(id, AccountType.Twitch)))
                .Where(a => a.IsMerged).GroupBy(a => a.MergedUserID).Select(g => g.First()).ToArray();
            var allUserRegions = UserRegion.MultiGetLocal(allAccounts.Select(a => new KeyInfo(a.MergedUserID))).ToDictionary(r => r.UserID);
            var allUsers = allUserRegions.Values.GroupBy(r => r.RegionID).SelectMany(g => User.MultiGet(g.Key, g.Select(r => new KeyInfo(r.UserID)))).ToDictionary(u => u.UserID);

            // Add Blocks
            var blockedAccounts = allAccounts.Where(a => blockedIDs.Contains(a.ExternalID)).ToArray();
            foreach (var block in blockedAccounts)
            {
                User otherUser;
                if (allUsers.TryGetValue(block.MergedUserID, out otherUser) && otherUser.IsMerged)
                {
                    UserBlock.ToggleBlock(user, otherUser, UserBlockStatus.Active);
                }
            }

            // Add Friends
            var friendAccounts = allAccounts.Where(a => friendIDs.Contains(a.ExternalID)).ToArray();
            foreach (var friend in friendAccounts)
            {
                UserRegion otherUserRegion;
                User otherUser;
                if (allUserRegions.TryGetValue(friend.MergedUserID, out otherUserRegion) && allUsers.TryGetValue(friend.MergedUserID, out otherUser) && otherUser.IsMerged)
                {
                    // Add Friend
                    Friendship.SystemCreateFriendship(user, userRegion, otherUser, otherUserRegion, FriendshipStatus.Confirmed, FriendshipStatus.Confirmed);
                }
            }

            // Add Friend Requests
            var requestedAccounts = allAccounts.Where(a => requestIDs.Contains(a.ExternalID)).ToArray();
            foreach (var friendRequest in requestedAccounts)
            {
                UserRegion otherUserRegion;
                User otherUser;
                if (allUserRegions.TryGetValue(friendRequest.MergedUserID, out otherUserRegion) && allUsers.TryGetValue(friendRequest.MergedUserID, out otherUser) && otherUser.IsMerged)
                {
                    // Add Friend
                    Friendship.SystemCreateFriendship(user, userRegion, otherUser, otherUserRegion, FriendshipStatus.AwaitingThem, FriendshipStatus.AwaitingMe);
                }
            }


            // Account Settings
            privacy.BlockStrangerPMs = privacy.BlockStrangerPMs || response.Value.AccountSettings.IgnoreWhispersFromStrangers;
            privacy.ShareActivityPrivacy = privacy.ShareActivityPrivacy == ShareActivityPrivacy.Hide || !response.Value.AccountSettings.ShareActivity
                ? ShareActivityPrivacy.Hide
                : ShareActivityPrivacy.Share;
            privacy.Update(p => p.BlockStrangerPMs, p => p.ShareActivityPrivacy);

            if (response.Value.AccountSettings.IsWhisperBanned)
            {
                UserBan.BanUser(user.UserID, UserBanType.Whisper);
            }

            if (response.Value.AccountSettings.IsInvisible)
            {
                var endpointProvider = new TwitchClientEndpointProvider(user.UserID);
                endpointProvider.UpdateSettings(userRegion, UserConnectionStatus.Invisible);
            }
        }
    }
}
