﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Net;
using System.Threading;
using Amazon.SQS.Model;
using Curse.Friends.TwitchInteropService.Configuration;
using Curse.Friends.TwitchInteropService.Stats;
using Curse.Logging;
using Newtonsoft.Json;
using System.Threading.Tasks;

namespace Curse.Friends.TwitchInteropService.Sqs
{
    abstract class BaseSqsConsumer
    {
        private readonly Thread[] _threads;
        private readonly string _queueUrl;

        protected readonly LogCategory Logger;
        protected readonly LogCategory DiagLogger;
        protected readonly LogCategory TraceLogger;

        private readonly CancellationTokenSource _cts = new CancellationTokenSource();
        private readonly TwitchInteropConnectionInfo _connectionInfo;
        private readonly System.Timers.Timer _loggingTimer;
        protected readonly SqsConsumerStats Stats;


#if CONFIG_DEBUG
        private const int MaxThreads = 1;
#elif CONFIG_STAGING
        private const int MaxThreads = 30;
#else
        private const int MaxThreads = 30;
#endif

        protected BaseSqsConsumer(TwitchInteropConnectionInfo connectionInfo, SqsConsumerStats stats, int numberOfThreads = 0)
        {
            Stats = stats;

            var typeName = GetType().Name;
            Logger = Logger = new LogCategory(typeName);
            DiagLogger = new LogCategory(typeName+"Diag") { Throttle = TimeSpan.FromMinutes(1), ReleaseLevel = LogLevel.Trace };
            TraceLogger = new LogCategory(typeName+"Trace") { Throttle = TimeSpan.FromMinutes(5), ReleaseLevel = LogLevel.Trace };

            _queueUrl = connectionInfo.StreamName;
            _connectionInfo = connectionInfo;

            if(numberOfThreads == 0)
            {
                numberOfThreads = MaxThreads;
            }

            var threads = new List<Thread>();
            for (var i = 0; i < numberOfThreads; i++)
            {
                var thread = new Thread(ConsumeQueue) { IsBackground = true };
                threads.Add(thread);
            }

            Logger.Info("Consumer created: " + connectionInfo.StreamName, new { numberOfThreads, MaxParallel });

            _loggingTimer = new System.Timers.Timer(TimeSpan.FromMinutes(1).TotalMilliseconds);
            _loggingTimer.Elapsed += timer_Elapsed;
            _threads = threads.ToArray();
        }

        void timer_Elapsed(object sender, System.Timers.ElapsedEventArgs e)
        {
            _loggingTimer.Enabled = false;

            try
            {
                LogQueueLength();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to query queue length");
            }
            finally
            {
                _loggingTimer.Enabled = true;
            }

        }

        public void Start()
        {
            // Try to get the queue length, for debugging purposes
            try
            {
                var attribs = SqsClient.CreateClient(_connectionInfo).Client.GetQueueAttributes(_queueUrl, new List<string> { "All" });
                Logger.Info($"Starting {_threads.Length} SQS consumers for {_queueUrl}", new { attribs.Attributes });
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to get SQS queue attributes.");
            }

            foreach (var thread in _threads)
            {
                thread.Start();
            }

            _loggingTimer.Start();
        }

        public void Stop()
        {
            _cts.Cancel();
            if (_loggingTimer != null)
            {
                _loggingTimer.Stop();
            }
        }


        private void LogQueueLength()
        {
            using (var client = SqsClient.CreateClient(_connectionInfo))
            {
                var attribs = client.Client.GetQueueAttributes(_queueUrl, new List<string> { "ApproximateNumberOfMessages" });

                Logger.Info("Current queue length: " + attribs.ApproximateNumberOfMessages, 
                    new
                    {
                        attribs.DelaySeconds,
                        attribs.ApproximateNumberOfMessages,
                        attribs.ApproximateNumberOfMessagesDelayed,
                        attribs.ApproximateNumberOfMessagesNotVisible,                        
                    });
            }
        }

#if CONFIG_DEBUG
        private const int MaxParallel = 1;
#else
        private const int MaxParallel = 8;
#endif

        private void ConsumeQueue()
        {

            var client = SqsClient.CreateClient(_connectionInfo);

            while (!_cts.IsCancellationRequested)
            {

                var recvMsgReq = new ReceiveMessageRequest
                {
                    QueueUrl = _queueUrl,
                    MaxNumberOfMessages = 10,
                    VisibilityTimeout = 30,
                    WaitTimeSeconds = 15,
                    AttributeNames = new List<string> { "All" },
                    MessageAttributeNames = new List<string> { "All" }
                };

                ReceiveMessageResponse res;

                var sw = new Stopwatch();
                sw.Start();

                try
                {
                    res = client.Client.ReceiveMessageAsync(recvMsgReq, _cts.Token).Result;
                }
                catch (AggregateException ex)
                {
                    if (ex.InnerExceptions.Any(e => e is OperationCanceledException))
                    {
                        break;
                    }
                    Logger.Warn(ex.InnerException, "Error reading SQS Queue");
                    continue;
                }
                catch (OperationCanceledException)
                {
                    break;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Error reading SQS Queue");
                    continue;
                }

                if (res.Messages == null)
                {
                    continue;
                }

                if (!res.Messages.Any())
                {
                    Thread.Sleep(50);
                    continue;
                }
                var newMessageElapsed = sw.ElapsedMilliseconds;

                sw.Restart();
                Parallel.ForEach(res.Messages, new ParallelOptions { MaxDegreeOfParallelism = MaxParallel }, message =>
                {
                    try
                    {
                        if (string.IsNullOrEmpty(message.Body))
                        {
                            StatsTracker.SqsConsumerStats.UnprocessableEvent();
                            Logger.Warn("Event's 'body' attribute is null or empty");
                            return;
                        }

                        var body = JsonConvert.DeserializeObject<SqsConsumedMessage>(message.Body);

                        if (body == null)
                        {
                            StatsTracker.SqsConsumerStats.UnprocessableEvent();
                            DiagLogger.Warn("Event's 'body' could not be deserialized.");
                            return;
                        }

                        ProcessMessage(body);
                    }
                    catch (Exception ex)
                    {
                        DiagLogger.Error(ex, "Failed to construct or process event contract.");
                        StatsTracker.SqsConsumerStats.UnprocessableEvent();
                    }
                });
                sw.Stop();
                var messageProcessTime = sw.ElapsedMilliseconds;
                sw.Restart(); 

                try
                {
                    // TODO: Don't delete failed events to ensure retry in the future?
                    var del = new DeleteMessageBatchRequest(_queueUrl, res.Messages.Select(m => new DeleteMessageBatchRequestEntry(m.MessageId, m.ReceiptHandle)).ToList());
                    var delRes = client.Client.DeleteMessageBatchAsync(del, _cts.Token).Result;

                    if (delRes.HttpStatusCode != HttpStatusCode.OK)
                    {
                        Logger.Warn("Failed to delete message(s), will be picked up after the visibility timeout expires", new { delRes.Failed, delRes.Successful });
                    }
                }
                catch (AggregateException ex)
                {
                    if (ex.InnerExceptions.Any(e => e is OperationCanceledException))
                    {
                        break;
                    }
                    Logger.Warn(ex.InnerException, "Failed to delete message(s), they will be picked up after the visibility timeout expires");
                }
                catch (OperationCanceledException)
                {
                    break;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to delete message(s), they will be picked up after the visibility timeout expires");
                }
                
                sw.Stop();
                var deleteTime = sw.ElapsedMilliseconds;

                DiagLogger.Trace("Message Processing Time", new
                {
                    newMessageElapsed,
                    messageProcessTime,
                    deleteTime,
                    MessageCount = res.Messages.Count
                });
            }

            client.Dispose();
        }

        protected abstract void ProcessMessage(SqsConsumedMessage wrapper);
    }
}
