﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using Curse.Friends.TwitchInterop;
using Curse.Friends.TwitchInteropService.Configuration;
using Curse.Friends.TwitchInteropService.Stats;
using Curse.Logging;
using Newtonsoft.Json;

namespace Curse.Friends.TwitchInteropService.Sqs.Processors
{
    class InteropConsumer : BaseSqsConsumer
    {
        private static readonly Dictionary<string, Type> _interopContractTypes = new Dictionary<string, Type>();
        private static readonly HashSet<string> ignoredEvents = new HashSet<string> { "friend_request_accepted", "set_presence_status", "set_presence_settings" };

        static InteropConsumer()
        {
            try
            {
                // Get all interop event types
                var contractTypes = typeof(BaseTwitchInteropContract).Assembly.GetTypes()
                    .Where(p => typeof(BaseTwitchInteropContract).IsAssignableFrom(p) && !p.IsAbstract && p.BaseType != null && p.BaseType.IsGenericType && p.GetCustomAttribute<TwitchInteropContractAttribute>() != null)
                    .ToDictionary(p => p.GetCustomAttribute<TwitchInteropContractAttribute>().Name, p => p.BaseType);

                foreach (var kvp in contractTypes)
                {
                    var genericArgs = kvp.Value.GetGenericArguments();
                    var contractType = genericArgs[0];

                    try
                    {
                        _interopContractTypes.Add(kvp.Key, contractType);
                    }
                    catch (ArgumentException)
                    {
                        Logging.Logger.Warn("Failed to process contract type, as it is a duplicate: " + kvp.Key);
                    }
                }
            }
            catch (Exception ex)
            {
                Logging.Logger.Fatal(ex, "Failed to initialize!");
            }
        }

        public InteropConsumer(TwitchInteropConnectionInfo connectionInfo, int numberOfThreads = 0) : base(connectionInfo, StatsTracker.SqsConsumerStats, numberOfThreads)
        {

        }

        protected override void ProcessMessage(SqsConsumedMessage wrapper)
        {
            BaseTwitchInteropContract contract;

            try
            {
                MessageAttribute attribute;
                if (!wrapper.MessageAttributes.TryGetValue("event", out attribute))
                {
                    Stats.UnprocessableEvent();
                    DiagLogger.Warn("Event did not have an 'event' message attribute", wrapper.TopicArn);
                    return;
                }
                var eventType = attribute.Value as string;
                if (eventType == null)
                {
                    Stats.UnprocessableEvent();
                    DiagLogger.Warn("Event's 'event' attribute was not a string", attribute);
                    return;
                }

                if (ignoredEvents.Contains(eventType))
                {
                    Stats.EventSkipped();
                    // these are expected to be ignored so we don't want to pollute the logs with unnecessary information.
                    return;
                }

                MessageAttribute originAttribute;
                if (!wrapper.MessageAttributes.TryGetValue("origin", out originAttribute))
                {
                    Stats.UnprocessableEvent();
                    DiagLogger.Warn("Event did not have an 'origin' message attribute", eventType);
                    return;
                }

                if (originAttribute.Value.ToString().Equals("curse"))
                {
                    Stats.UnprocessableEvent();
                    return;
                }

                Type contractType;
                if (!_interopContractTypes.TryGetValue(eventType, out contractType))
                {
                    Stats.UnprocessableEvent();
                    TraceLogger.Warn("Unknown contract type: " + eventType);
                    return;
                }

                contract = (BaseTwitchInteropContract) JsonConvert.DeserializeObject(wrapper.Message, contractType);
            }
            catch (Exception ex)
            {
                DiagLogger.Error(ex, "Failed to deserialize or construct event contract.");
                Stats.UnprocessableEvent();
                return;
            }

            try
            {
                if (TwitchInteropConfiguration.Instance.ConsumerEnabled)
                {
                    LogEvent(TraceLogger, LogLevel.Trace, "Consuming inbound event: " + contract.GetContractName(), contract);
                    if (contract.Validate())
                    {
                        // whitelisting certain twitch users. 
                        if (TwitchInteropConfiguration.Instance.UseWhitelist && !contract.IsWhitelisted())
                        {
                            LogEvent(DiagLogger, LogLevel.Trace, "Skipping! Non whitelisted users found: " + contract.GetContractName(), contract);
                            Stats.EventSkipped();
                            return;
                        }

                        contract.ProcessInbound();
                        Stats.EventProcessed();
                    }
                    else
                    {
                        Stats.EventFailed();
                        contract.LogEvent("Data validation failed for inbound event.");
                    }
                }
                else
                {
                    Stats.EventSkipped();
                    LogEvent(TraceLogger, LogLevel.Trace, "Consumer disabled. Skipping inbound event: " + contract.GetContractName(), contract);
                }
            }
            catch (NotImplementedException)
            {
                DiagLogger.Debug(string.Format("Not yet processing {0}", contract.GetContractName()));
                Stats.EventFailed();
            }
            catch (Exception ex)
            {
                LogEvent(DiagLogger, LogLevel.Error, "Failed to process an event: " + contract.GetContractName(), contract, ex);
                Stats.EventFailed();
            }


        }

        private void LogEvent(LogCategory logger, LogLevel level, string message, BaseTwitchInteropContract e, Exception ex = null)
        {
            logger.Log(level, DateTime.UtcNow, ex, message, e.GetLogData());
        }
    }
}
