﻿using System;
using Curse.Extensions;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.TwitchInteropService.Configuration;
using Curse.Friends.TwitchInteropService.Sqs.Processors;
using Curse.Friends.TwitchInteropService.Stats;
using Newtonsoft.Json;
using System.Linq;
using Curse.Aerospike;

namespace Curse.Friends.TwitchInteropService.Sqs.StreamStatus
{
    class StreamDownConsumer : BaseSqsConsumer
    {
        private const string _streamDownEventName = "stream_change_down";

        public StreamDownConsumer(TwitchInteropConnectionInfo connectionInfo, int numberOfThreads = 0) : base(connectionInfo, StatsTracker.SqsStreamDownStats, numberOfThreads)
        {

        }

        protected override void ProcessMessage(SqsConsumedMessage wrapper)
        {
            MessageAttribute attribute;
            if (!wrapper.MessageAttributes.TryGetValue("event", out attribute))
            {
                Stats.UnprocessableEvent();
                DiagLogger.Warn("Event did not have an 'event' message attribute", wrapper.TopicArn);
                return;
            }
            var eventType = attribute.Value as string;
            if (eventType == null)
            {
                Stats.UnprocessableEvent();
                DiagLogger.Warn("Event's 'event' attribute was not a string", attribute);
                return;
            }

            if (eventType != _streamDownEventName)
            {
                Stats.EventSkipped();
                return;
            }

            var body = JsonConvert.DeserializeObject<StreamChangeEvent>(wrapper.Message);
            if (body == null || body.ChannelID == 0)
            {
                Stats.UnprocessableEvent();
                return;
            }

            if (!TwitchInteropConfiguration.Instance.StreamQueuesEnabled)
            {
                Stats.EventSkipped();
                TraceLogger.Info("Stream consumers disabled, skipping event");
                return;
            }

            ExternalCommunity community;
            try
            {
                community = TwitchModelHelper.GetOrCreateCommunity(body.ChannelID.ToString());
                if (!community.IsLive)
                {
                    Stats.EventSkipped();
                    return;
                }
            }
            catch (Exception ex)
            {
                DiagLogger.Warn(ex, "Unable to fetch ExternalCommunity, skipping event", body);
                Stats.EventFailed();
                return;
            }

            community.IsLive = false;
            community.LiveTimestamp = DateTime.UtcNow.ToEpochMilliseconds();
            community.Update(c => c.IsLive, c => c.LiveTimestamp);

            if (community.MappedGroups != null && community.MappedGroups.Count > 0)
            {
                var groups = Group.MultiGetLocal(community.MappedGroups.Select(id => new KeyInfo(id)));
                foreach (var group in groups)
                {
                    group.EnsureWritable().UpdateIsStreaming(community);
                }
            }

            ExternalCommunityLinkChangedCoordinator.Create(community, ExternalCommunityLinkChangeType.LiveStatus);
            ExternalCommunityIndexWorker.Create(community);
            TraceLogger.Debug("Detecting stream going down", new {community.ExternalName, community.ExternalID});
            Stats.EventProcessed();
        }
    }
}
