﻿using System;
using System.Threading;
using Amazon;
using Amazon.Runtime;
using Amazon.SecurityToken;
using Amazon.SecurityToken.Model;
using Amazon.SQS;
using Curse.Friends.TwitchInteropService.Configuration;

namespace Curse.Friends.TwitchInteropService.Sqs
{
    public class SqsClient : IDisposable
    {
        private readonly AmazonSQSClient _client;
        private readonly IDisposable[] _disposables;

        private SqsClient(AmazonSQSClient client, params IDisposable[] disposables)
        {
            _client = client;
            _disposables = disposables ?? new IDisposable[0];
        }

        public AmazonSQSClient Client { get { return _client; } }

        private static SqsClient CreateAssumeRoleClient(string accessID, string secret, string roleArn, string regionEndpointName)
        {
            var stsClient = new AmazonSecurityTokenServiceClient(new BasicAWSCredentials(accessID, secret), RegionEndpoint.USEast1);
            var assumeRoleRequest = new AssumeRoleRequest { RoleArn = roleArn, RoleSessionName = string.Format("{0}_consumer", Environment.MachineName) };
            var credentials = new STSAssumeRoleAWSCredentials(stsClient, assumeRoleRequest);
            var client = new AmazonSQSClient(credentials, RegionEndpoint.GetBySystemName(regionEndpointName));
            return new SqsClient(client, credentials, stsClient);
        }

        private static SqsClient CreateInstanceProfileClient(string roleArn, string regionEndpointName)
        {
            var stsClient = new AmazonSecurityTokenServiceClient(new InstanceProfileAWSCredentials(), RegionEndpoint.GetBySystemName(regionEndpointName));
            var assumeRoleRequest = new AssumeRoleRequest { RoleArn = roleArn, RoleSessionName = string.Format("{0}_consumer", Environment.MachineName) };
            var credentials = new STSAssumeRoleAWSCredentials(stsClient, assumeRoleRequest);
            var client = new AmazonSQSClient(credentials, RegionEndpoint.GetBySystemName(regionEndpointName));
            return new SqsClient(client, credentials, stsClient);
        }

        private static SqsClient CreateInstanceProfileClient(string regionEndpointName)
        {
            var client = new AmazonSQSClient(RegionEndpoint.GetBySystemName(regionEndpointName));
            return new SqsClient(client);
        }

        private static SqsClient CreateBasicClient(string accessID, string secret, string regionEndpointName)
        {
            var credentials = new BasicAWSCredentials(accessID, secret);
            var client = new AmazonSQSClient(credentials, RegionEndpoint.GetBySystemName(regionEndpointName));
            return new SqsClient(client);
        }

        public static SqsClient CreateClient(TwitchInteropConnectionInfo connectionInfo)
        {
            switch (connectionInfo.CredentialType)
            {
                case AwsCredentialType.AssumeRole:
                    return CreateAssumeRoleClient(connectionInfo.UserAccessKey, connectionInfo.UserAccessSecret, connectionInfo.RoleArn, connectionInfo.RegionEndpoint);
                case AwsCredentialType.InstanceProfile:
                    return string.IsNullOrEmpty(connectionInfo.RoleArn) 
                        ? CreateInstanceProfileClient(connectionInfo.RegionEndpoint)
                        : CreateInstanceProfileClient(connectionInfo.RoleArn, connectionInfo.RegionEndpoint);
                default:
                    return CreateBasicClient(connectionInfo.UserAccessKey, connectionInfo.UserAccessSecret, connectionInfo.RegionEndpoint);
            }
        }     

        
        private bool _isDisposed;
        private readonly object _disposeLock = new object();

        public void Dispose()
        {
            var acquired = Monitor.TryEnter(_disposeLock, TimeSpan.FromSeconds(5));

            if (!acquired)
            {
                return;
            }

            if (_isDisposed)
            {
                return;
            }

            try
            {
                _isDisposed = true;

                _client.Dispose();
                foreach (var disposable in _disposables)
                {
                    disposable.Dispose();
                }

            }
            finally
            {
                Monitor.Exit(_disposeLock);
            }
        }
    }
}
