﻿using System;
using System.Collections.Generic;
using System.Net;
using Amazon.SQS.Model;
using Curse.Friends.TwitchInteropService.Configuration;
using Curse.Friends.TwitchInteropService.Stats;
using Curse.Logging;
using Newtonsoft.Json;

namespace Curse.Friends.TwitchInteropService.Sqs
{
    public class SqsPublisher
    {
        private readonly SqsClient _client;
        private readonly string _queueUrl;
        private readonly LogCategory Logger = new LogCategory("SQS Producer");

        private SqsPublisher(string queueUrl, SqsClient client)
        {
            _queueUrl = queueUrl;
            _client = client;
        }

        public static SqsPublisher Create(TwitchInteropConnectionInfo connectionInfo)
        {
            var client = SqsClient.CreateClient(connectionInfo);
            return new SqsPublisher(connectionInfo.StreamName, client);
        }

        public void SendMessage(object message)
        {
            try
            {
                // TODO: Buffer these and group them in batches instead of sending each one (SendMessageBatch) - max size is 256kb
                var request = new SendMessageRequest(_queueUrl, JsonConvert.SerializeObject(message));
                var resp = _client.Client.SendMessage(request);
                if (resp.HttpStatusCode == HttpStatusCode.OK)
                {
                    StatsTracker.SqsPublisherStats.EventSent();
                }
                else
                {
                    StatsTracker.SqsPublisherStats.EventFailed();
                    Logger.Warn("Failed to send a message to SQS", message);
                }
            }
            catch (Exception ex)
            {
                StatsTracker.SqsPublisherStats.EventFailed();
                Logger.Error(ex, "Failed to send a message to SQS", message);
            }
        }

        public void SendEvent(string eventType, string serializedMessage)
        {
            try
            {
                // TODO: Buffer these and group them in batches instead of sending each one (SendMessageBatch) - max size is 256kb
                var request = new SendMessageRequest(_queueUrl, serializedMessage)
                {
                    MessageAttributes = new Dictionary<string, MessageAttributeValue> 
                    {
                        { "event", new MessageAttributeValue { DataType = "String", StringValue =  eventType } },
                        { "origin", new MessageAttributeValue { DataType = "String", StringValue =  "curse" } }
                    }
                };
                

                var resp = _client.Client.SendMessage(request);
                if (resp.HttpStatusCode == HttpStatusCode.OK)
                {
                    StatsTracker.SqsPublisherStats.EventSent();
                }
                else
                {
                    StatsTracker.SqsPublisherStats.EventFailed();
                    Logger.Warn("Failed to send a message to SQS: " + eventType, serializedMessage);
                }
            }
            catch (Exception ex)
            {
                StatsTracker.SqsPublisherStats.EventFailed();
                Logger.Error(ex, "Failed to send a message to SQS: " + eventType, serializedMessage);
            }
        }

    }
}
