﻿using System;
using Curse.CloudServices.Jobs;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.Queues;
using Curse.Friends.ServerHosting;
using Curse.Friends.TwitchApi;
using Curse.Friends.TwitchInteropService.Kinesis;
using Curse.Friends.TwitchInteropService.Processors;
using Curse.Friends.TwitchInteropService.Sqs;
using Curse.Friends.TwitchInteropService.Stats;
using Curse.Logging;
using Curse.Friends.TwitchInteropService.UserReporting;
using Curse.Friends.UserEvents;
using Curse.Friends.AuthenticationClient;
using Curse.Friends.Data.Messaging;
using Curse.Friends.TwitchInteropService.Configuration;
using Curse.Friends.TwitchInterop;

namespace Curse.Friends.TwitchInteropService
{
    internal class TwitchInteropServer : ServerHost<KinesisHost, KinesisShardIterator>
    {
        private static readonly TwitchInteropServer _instance = new TwitchInteropServer();

        public static TwitchInteropServer Instance
        {
            get { return _instance; }
        }

        protected override void CustomStartup()
        {
            StatsTracker.Start();
            CheckTwitchApiAccess();

            try
            {
                AuthenticationProvider.Initialize(TwitchInteropConfiguration.Instance.AuthenticationServiceUrl, TwitchInteropConfiguration.Instance.AuthenticationServiceSiteID, TwitchInteropConfiguration.Instance.AuthenticationServiceSiteKey);
            }
            catch (Exception ex)
            {
                Logger.Fatal(ex, "Failed to initialize authentication provider.");
            }

            try
            {
                TwitchInteropEventWorkerProcessor.Initialize();
            }
            catch (Exception ex)
            {
                Logger.Fatal(ex, "Failed to start twitch interop event processor!");
            }

            if (StorageConfiguration.CurrentRegion.IsDefault && TwitchInteropConfiguration.Instance.KinesisEnabled)
            {

                Logger.Info("Starting Kinesis Manager...");

                try
                {
                    KinesisManager.Start();
                }
                catch (Exception ex)
                {
                    Logger.Fatal(ex, "Failed to start kinesis manager!");
                }

                try
                {
                    EnsureAllHosted();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to ensure all shards are hosted. We will retry in 10 seconds.");
                }

                AddTimer("EnsureAllHosted", TimeSpan.FromSeconds(10), EnsureAllHosted);
                AddTimer("RefreshKinesisShards", TimeSpan.FromSeconds(60), KinesisManager.RefreshKinesisShards);
            }
            else
            {
                Logger.Info("Kinesis streams will not be processed from this region.");
            }

            try
            {
                Logger.Info("Starting SQS Manager...");
                SqsManager.Start(StorageConfiguration.CurrentRegion.IsDefault, true);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to start SQS Manager");
            }


            JobScheduler.Initialize(StorageConfiguration.CurrentRegion.IsDefault);
            AddTimer("RefreshTwitchGameMapping", TimeSpan.FromHours(1), GameMapping.LoadData);

            UserReportWorker.StartProcessor(UserReportProcessor.Process);
            UserEventWorker.StartProcessor(TwitchInteropEventWorkerProcessor.Process);
            TwitchMergeWorker.StartProcessor(TwitchMergeWorkerProcessor.Process);
            TwitchCommunityModsWorker.StartProcessor(TwitchCommunityModsProcessor.Process);

            ChattersListProcessor.Initialize(FriendsServiceConfiguration.Instance.TwitchShimApiKey, FriendsServiceConfiguration.Instance.TwitchShimUrl);
            ChattersListWorker.StartProcessor(ChattersListProcessor.Process);
        }

        protected override void EnsureHosted(KinesisShardIterator hostable)
        {
            KinesisManager.HostShard(hostable);
        }

        protected override void EnsureUnhosted(KinesisShardIterator hostable)
        {
            KinesisManager.UnhostShard(hostable);
        }

        protected override void CustomStop()
        {
            KinesisManager.Stop();
            SqsManager.Stop();
            StatsTracker.Stop();
        }

        static void CheckTwitchApiAccess()
        {
            try
            {
                using (var client = new CurseShimClient(FriendsServiceConfiguration.Instance.TwitchShimApiKey, FriendsServiceConfiguration.Instance.TwitchShimUrl))
                {
                    var resp = client.CheckAccess();
                    if (resp.Status != TwitchResponseStatus.Success)
                    {
                        Logger.Fatal("Unable to successfully communicate with Twitch Shim API", resp);
                    }
                    else
                    {
                        Logger.Info("Confirmed access to the Twitch Shim API. ");
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to check Twitch API access.");
            }

        }

    }
}
