﻿using Curse.Friends.Data;
using Curse.Friends.Tracing;
using Curse.Friends.TwitchInteropService.Stats;
using Curse.Logging;
using System;
using System.Collections.Generic;
using System.Linq.Expressions;

namespace Curse.Friends.TwitchInteropService.UserMutation
{
    public class UserMutationProcessor
    {
        private static readonly LogCategory Logger = new LogCategory("UserMutationProcessor");
        private static readonly FilteredUserLogger FilteredLogger = new FilteredUserLogger("UserMutationProcessor");
        private static readonly LogCategory ThrottledLogger = new LogCategory("UserMutationProcessor") { Throttle = TimeSpan.FromMinutes(1), ReleaseLevel = LogLevel.Debug };
      
        public static void Process(UserMutationEvent identityEvent)
        {
            switch (identityEvent.Type)
            {
                case "user_login_change":
                    ProcessLoginNameChange(identityEvent);
                    break;
                case "displayname_change":
                    ProcessDisplayNameChange(identityEvent);
                    break;
                    
                default:
                    Logger.Warn($"Unknown Identity Event Type: {identityEvent.Type}! Assuming login name change", new { identityEvent });
                    ProcessLoginNameChange(identityEvent);                   
                    break;
            }
        }

        public static void ProcessDisplayNameChange(UserMutationEvent userEvent)
        {
            // Update community if we track it
            UpdateCommunity(ExternalCommunity.GetByTwitchID(userEvent.UserID), userEvent.Data.Changes);

            // Get this user from the local database
            var externalAccount = ExternalAccount.GetByTwitchUserID(userEvent.UserID);

            // If we don't have an external account, this event can be ignored
            if (externalAccount == null)
            {
                FilteredLogger.Log(null, null, "Received a user mutation event for a user that is not in our local database.", userEvent);
                StatsTracker.UserMutationStreamStats.EventSkipped();
                return;
            }

            // Get the local user record (this will be null for unmerged)
            var user = externalAccount.GetMergedUser();

            var changes = userEvent.Data.Changes;

            // Do some basic data validation
            if (string.IsNullOrEmpty(changes.DisplayName))
            {
                FilteredLogger.Log(user, externalAccount, "Received a display name change event for a user that is not in our local database.", userEvent);
                StatsTracker.UserMutationStreamStats.EventSkipped();
                return;
            }            

            // Update the external account identity details            
            externalAccount.ExternalDisplayName = changes.DisplayName;
            externalAccount.Update(p => p.ExternalDisplayName);
            StatsTracker.UserMutationStreamStats.EventProcessed();

            // If the external account is not yet merged, no need to continue
            if (!externalAccount.IsMerged)
            {
                FilteredLogger.Log(user, externalAccount, "Received a user mutation event for an unmerged external account.", new { userEvent, changes });
                return;
            }

            if (user == null)
            {
                FilteredLogger.Log(null, externalAccount, "Received a user mutation event that mapped to a merged user, but the user record could not be retrieved from the database.", new { userEvent, changes });
                return;
            }

            FilteredLogger.Log(user, externalAccount, "Processing display name change event", new { userEvent, changes });
            user.ChangeUsername(user.Username, changes.DisplayName);

        }

        public static void ProcessLoginNameChange(UserMutationEvent userEvent)
        {
            // Update community if we track it
            UpdateCommunity(ExternalCommunity.GetByTwitchID(userEvent.UserID), userEvent.Data.Changes);

            // Get this user from the local database
            var externalAccount = ExternalAccount.GetByTwitchUserID(userEvent.UserID);

            // If we don't have an external account, this event can be ignored
            if (externalAccount == null)
            {
                FilteredLogger.Log(null, null, "Received a user mutation event for a user that is not in our local database.", userEvent);
                StatsTracker.UserMutationStreamStats.EventSkipped();
                return;
            }

            // Get the local user record (this will be null for unmerged)
            var user = externalAccount.GetMergedUser();

            var changes = userEvent.Data.Changes;

            // Do some basic data validation
            if (string.IsNullOrEmpty(changes.Login))
            {
                FilteredLogger.Log(user, externalAccount, "Received a user mutation event for a user that is not in our local database.", userEvent);
                StatsTracker.UserMutationStreamStats.EventSkipped();
                return;
            }

            // Coalesce the display name to the login, if necessary
            if (string.IsNullOrEmpty(changes.DisplayName))
            {
                FilteredLogger.Log(user, externalAccount, "Received a user mutation event with a null or empty display name. The login name will be used instead.", userEvent);
                changes.DisplayName = changes.Login;
            }
            
            // Update the external account identity details
            externalAccount.ExternalUsername = changes.Login;
            externalAccount.ExternalDisplayName = changes.DisplayName;
            externalAccount.Update(p => p.ExternalUsername, p => p.ExternalDisplayName);
            StatsTracker.UserMutationStreamStats.EventProcessed();

            // If the external account is not yet merged, no need to continue
            if (!externalAccount.IsMerged)
            {
                FilteredLogger.Log(user, externalAccount, "Received a user mutation event for an unmerged external account.", new {  userEvent, changes });                
                return;
            }
            
            if (user == null)
            {
                FilteredLogger.Log(null, externalAccount, "Received a user mutation event that mapped to a merged user, but the user record could not be retrieved from the database.", new { userEvent, changes });                
                return;
            }

            FilteredLogger.Log(user, externalAccount, "Processing user mutation event", new { userEvent, changes });
            user.ChangeUsername(changes.Login, changes.DisplayName);

        }

        private static void UpdateCommunity(ExternalCommunity community, UserMutationChanges changes)
        {
            try
            {
                if (community == null)
                {
                    return;
                }

                ThrottledLogger.Debug("Renaming community from rename event", new { community = community.GetLogData(), changes });

                var updates = new List<Expression<Func<ExternalCommunity, object>>>();
                if (!string.IsNullOrEmpty(changes.Login) && community.ExternalName != changes.Login)
                {
                    community.ExternalName = changes.Login;
                    updates.Add(c => c.ExternalName);
                }

                if (!string.IsNullOrEmpty(changes.DisplayName) && community.ExternalDisplayName != changes.DisplayName)
                {
                    community.ExternalDisplayName = changes.DisplayName;
                    updates.Add(c => c.ExternalDisplayName);
                }

                if (updates.Count > 0)
                {
                    community.Update(updates.ToArray());
                }
            }
            catch(Exception ex)
            {
                ThrottledLogger.Warn(ex, "Failed to update external community on rename", new { Community = community?.GetLogData(), changes });
            }
        }
    }
}
