﻿using System;
using System.Collections.Generic;
using Amazon.MachineLearning;
using Curse.Extensions;
using Curse.Friends.Configuration;
using Curse.Friends.Data.Queues;
using Curse.Logging;
using Curse.Friends.Enums;
using Curse.Friends.TwitchApi;
using User = Curse.Friends.Data.User;

namespace Curse.Friends.TwitchInteropService.UserReporting
{
    public static class UserReportProcessor
    {
        private static readonly LogCategory Logger = new LogCategory("UserReportProcessor") { AlphaLevel = LogLevel.Trace };
        private static readonly Curse.Friends.TwitchApi.CurseShimClient _twitchApi;

        static UserReportProcessor()
        {
            try
            {
                _twitchApi = new Curse.Friends.TwitchApi.CurseShimClient(FriendsServiceConfiguration.Instance.TwitchShimApiKey, FriendsServiceConfiguration.Instance.TwitchShimUrl);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to create the Twitch shim api client.");
            }

            try
            {
                ReportTypeMap = new Dictionary<UserReportType, string>
                {
                    { UserReportType.Whisper, "curse_whisper" },
                    { UserReportType.PrivateGroup, "curse_private_group" },
                    { UserReportType.PublicGroup, "curse_public_group" }
                };
            }
            catch (Exception ex)
            {
                Logger.Fatal(ex, "Failed to create the report type map during initialization.");
            }

            try
            {
                ReportReasonMap = new Dictionary<UserReportReason, string>
                {
                    { UserReportReason.BitViolation, "bits_violation" },
                    { UserReportReason.ChatBanEvasion, "chat_ban_evasion" },
                    { UserReportReason.Cheating, "cheating" },
                    { UserReportReason.CreativeConduct, "creative_conduct" },
                    { UserReportReason.Gore, "gore" },
                    { UserReportReason.Harassment, "harassment" },
                    { UserReportReason.Harm, "harm" },
                    { UserReportReason.HateSpeech, "hate_speech" },
                    { UserReportReason.Impersonation, "impersonation" },
                    { UserReportReason.MusicConduct, "music_conduct" },
                    { UserReportReason.Nongaming, "nongaming" },
                    { UserReportReason.OffensiveUsername, "offensive_username" },
                    { UserReportReason.Other, "other" },
                    { UserReportReason.Porn, "porn" },
                    { UserReportReason.Prohibited, "prohibited" },
                    { UserReportReason.SelfHarm, "selfharm" },
                    { UserReportReason.SocialEatingViolation, "social_eating_violation" },
                    { UserReportReason.Spam, "spam" },
                    { UserReportReason.TosBanEvasion, "tos_ban_evasion" },
                    { UserReportReason.Underaged, "underaged" },
                };
            }
            catch (Exception ex)
            {

                Logger.Fatal(ex, "Failed to create the report reason map during initialization.");
            }

            
        }

        private static readonly Dictionary<UserReportType, string> ReportTypeMap;

        private static readonly Dictionary<UserReportReason, string> ReportReasonMap;

        public static void Process(UserReportWorker worker)
        {
            // Look up the Twitch IDs for both users
            var sourceUser = User.FindByUserID(worker.SourceUserID);
            var targetUser = User.FindByUserID(worker.TargetUserID);

            if (sourceUser == null || targetUser == null)
            {
                Logger.Info("Skipping report. Unable to retrieve a user");
                return;
            }

            if (!sourceUser.IsMerged)
            {
                Logger.Info("Skipping report. The source user account has not been merged.");
                return;
            }

            if (!targetUser.IsMerged)
            {
                Logger.Info("Skipping report. The target user account has not been merged.");
                return;
            }

            var type = ReportTypeMap.GetValueOrDefault(worker.Type);
            var reason = ReportReasonMap.GetValueOrDefault(worker.Reason);
            var resp = _twitchApi.CreateReport(sourceUser.TwitchID, targetUser.TwitchID, type, reason, worker.Description);
            if (resp.Status != TwitchResponseStatus.Success)
            {
                Logger.Warn("Failed to submit user report to Twitch", resp);
            }
            else
            {
                Logger.Debug("Successfully submitted user report to Twitch!", worker);
            }
        }
    }
}
