﻿using System;
using System.Collections.Concurrent;
using System.Threading;
using System.Threading.Tasks;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.ServiceClients;
using Curse.Logging;
using Curse.Aerospike;
using System.Linq;
using System.Collections.Generic;

namespace Curse.Friends.TwitchService
{
    public static class AvatarUpdatePump
    {
        private class AvatarUpdate
        {
            public AvatarType Type { get; set; }

            public string EntityID { get; set; }

            public string Url { get; set; }

            public DateTime Timestamp { get; set; }
        }

        private static readonly CancellationTokenSource _cts = new CancellationTokenSource();
        private static ConcurrentBag<AvatarUpdate> _updateBatch = new ConcurrentBag<AvatarUpdate>();

        public static void UpdateAvatar(AvatarType type, string entityID, string url)
        {
            var update = new AvatarUpdate
            {
                Type = type,
                EntityID = entityID,
                Url = url,
                Timestamp = DateTime.UtcNow,
            };

            _updateBatch.Add(update);
        }

        public static void Start()
        {
            Task.Factory.StartNew(PumpUpdates, TaskCreationOptions.LongRunning);
        }

        public static void Stop()
        {
            _cts.Cancel();
            ProcessUpdates(_updateBatch);
        }

        private static void PumpUpdates()
        {
            while (!_cts.IsCancellationRequested)
            {
                AvatarUpdate update = null;
                try
                {
                    _cts.Token.WaitHandle.WaitOne(5000);

                    var updates = _updateBatch;
                    _updateBatch = new ConcurrentBag<AvatarUpdate>();
                    ProcessUpdates(updates);
                }
                catch (OperationCanceledException)
                {
                    break;
                }
                catch (Exception ex)
                {
                    Logger.Warn(ex, "Error updating avatar", new {Avatar = update});
                }
            }
        }

        private static void ProcessUpdates(ConcurrentBag<AvatarUpdate> avatarUpdates)
        {
            try
            {
                var updates = new Dictionary<Tuple<int, string>, AvatarUpdate>();
                foreach(var update in avatarUpdates)
                {
                    var key = Tuple.Create((int)update.Type, update.EntityID);
                    AvatarUpdate existing;
                    if(!updates.TryGetValue(key, out existing) || existing.Timestamp > update.Timestamp)
                    {
                        updates[key] = update;
                    }
                }

                if(updates.Count == 0)
                {
                    return;
                }

                var avatars = Avatar.MultiGetLocal(updates.Select(kvp => new KeyInfo(kvp.Key.Item1, kvp.Key.Item2))).ToDictionary(a=>Tuple.Create(a.AvatarType, a.EntityID));
                var requests = new List<Curse.ServiceClients.Contracts.BulkUpdateAvatarUrlsUpdate>();
                foreach(var kvp in updates)
                {
                    Avatar existing;
                    if(!avatars.TryGetValue(kvp.Key, out existing) || existing.Url != kvp.Value.Url)
                    {
                        requests.Add(new Curse.ServiceClients.Contracts.BulkUpdateAvatarUrlsUpdate
                        {
                            Type = GetContractType(kvp.Value.Type),
                            EntityID = kvp.Value.EntityID,
                            Url = kvp.Value.Url,
                        });
                    }
                }

                if(requests.Count > 0)
                {
                    var response = FriendsServiceClients.Instance.AvatarAdmin.BulkUpdateAvatarUrls(new Curse.ServiceClients.Contracts.BulkUpdateAvatarUrlsRequest
                    {
                        Updates = requests.ToArray()
                    });
                    if(!response.Success)
                    {
                        Logger.Warn("Failed to bulk update avatars", new { Requests = requests });
                    }
                }
            }
            catch(Exception ex)
            {
                Logger.Warn(ex, "Error updating avatars");
            }
        }

        private static Curse.ServiceClients.Contracts.AvatarType GetContractType(AvatarType type)
        {
            switch (type)
            {
                case AvatarType.Group:
                    return Curse.ServiceClients.Contracts.AvatarType.Group;
                case AvatarType.GroupCover:
                    return Curse.ServiceClients.Contracts.AvatarType.GroupCover;
                case AvatarType.GroupEmoticon:
                    return Curse.ServiceClients.Contracts.AvatarType.GroupEmoticon;
                case AvatarType.SyncedAccount:
                    return Curse.ServiceClients.Contracts.AvatarType.SyncedAccount;
                case AvatarType.SyncedCommunity:
                    return Curse.ServiceClients.Contracts.AvatarType.SyncedCommunity;
                case AvatarType.SyncedEmoticon:
                    return Curse.ServiceClients.Contracts.AvatarType.SyncedEmoticon;
                case AvatarType.TwitchChatBadge:
                    return Curse.ServiceClients.Contracts.AvatarType.TwitchChatBadge;
                case AvatarType.TwitchEmote:
                    return Curse.ServiceClients.Contracts.AvatarType.TwitchEmote;
                case AvatarType.User:
                    return Curse.ServiceClients.Contracts.AvatarType.User;
                default:
                    throw new InvalidOperationException("Unsupported Avatar Type: " + type);
            }
        }

    }
}
