﻿using System;
using System.Collections.Concurrent;
using System.IO;
using System.Net;
using System.Threading;
using Curse.Logging;
using Newtonsoft.Json;

namespace Curse.Friends.TwitchService.Chat.Firehose
{
    public class FirehoseManager
    {
        private static readonly FirehoseManager _instance = new FirehoseManager();
        public static FirehoseManager Instance { get { return _instance; } }

        private readonly ConcurrentDictionary<string, string> _channelMap = new ConcurrentDictionary<string, string>();

        public event EventHandler<FirehoseMessageEventArgs> MessageReceived; 

        public void Open()
        {
            new Thread(Run) {IsBackground = true}.Start();
        }

        public void Close()
        {
            _isRunning = false;
        }

        private bool _isRunning = true;

        private void Run()
        {
            while (_isRunning)
            {
                try
                {
                    var request = WebRequest.Create(new Uri("http://tmi.twitch.tv/firehose?oauth_token=1dhfluda1463598xplyvomj9oust56&login=curseappbot"));
                    ((HttpWebRequest) request).AllowReadStreamBuffering = false;

                    using (var response = request.GetResponse())
                    {
                        var stream = response.GetResponseStream();
                        if (stream == null)
                        {
                            Thread.Sleep(100);
                            continue;
                        }
                        using (var streamReader = new StreamReader(stream))
                        {
                            ReadUntilDisconnected(streamReader);
                        }
                    }
                }
                catch (Exception ex)
                {
                    Logger.Warn(ex, "Error while reading firehose");
                }
            }
        }

        private void ReadUntilDisconnected(StreamReader reader)
        {
            try
            {
                FirehoseEvent firehoseEvent = null;

                while (_isRunning)
                {
                    var line = reader.ReadLine();
                    if (line == null)
                    {
                        return;
                    }

                    if (string.IsNullOrEmpty(line) || line == ":ping")
                    {
                        continue;
                    }

                    if (line.StartsWith("event:"))
                    {
                        firehoseEvent = new FirehoseEvent
                        {
                            EventName = line.Replace("event:", string.Empty)
                        };
                    }
                    else if (line.StartsWith("data:"))
                    {
                        if (firehoseEvent == null)
                        {
                            Logger.Debug("Data received without an event line");
                            continue;
                        }

                        firehoseEvent.Data = JsonConvert.DeserializeObject<FirehoseMessage>(line.Replace("data:", string.Empty));
                        ProcessEvent(firehoseEvent);
                        firehoseEvent = null;
                    }
                }
            }
            catch (IOException ex)
            {
                Logger.Debug(ex, "IO Error while reading firehose");
            }
            catch (JsonException ex)
            {
                Logger.Debug(ex, "JSON Error while reading firehose");
            }
        }

        private void ProcessEvent(FirehoseEvent evt)
        {
            string externalID;
            if (!_channelMap.TryGetValue(evt.Data.Room, out externalID) && !_channelMap.TryGetValue(evt.Data.Target, out externalID))
            {
                return;
            }

            if (string.IsNullOrEmpty(evt.Data.Command))
            {
                evt.Data.Command = evt.EventName.Trim();
            }

            var message = FirehoseMessageParser.ParseMessage(evt.Data);
            if (message != null && MessageReceived != null)
            {
                MessageReceived.BeginInvoke(this, new FirehoseMessageEventArgs {ExternalID = externalID, Message = message}, EndMessageReceivedInvoke, null);
            }
        }

        private void EndMessageReceivedInvoke(IAsyncResult result)
        {
            var res = (System.Runtime.Remoting.Messaging.AsyncResult) result;
            var invokedMethod = (EventHandler<FirehoseMessageEventArgs>) res.AsyncDelegate;

            try
            {
                invokedMethod.EndInvoke(result);
            }
            catch (Exception ex)
            {
                Logger.Debug(ex, "Exception thrown ending event handler invoke");
            }
        }

        public void RegisterStream(string channelName, string externalID)
        {
            if (!channelName.StartsWith("#"))
            {
                channelName = "#" + channelName;
            }
            _channelMap.TryAdd(channelName, externalID);
        }

        public void UnregisterStream(string channelName)
        {
            if (!channelName.StartsWith("#"))
            {
                channelName = "#" + channelName;
            }
            string externalID;
            _channelMap.TryRemove(channelName, out externalID);
        }
    }
}
