﻿using System.Collections.Generic;
using Curse.Extensions;
using Curse.Friends.Enums;
using Curse.Friends.TwitchService.Chat.Parsing;

namespace Curse.Friends.TwitchService.Chat.Firehose
{
    public static class FirehoseMessageParser
    {
        public static TwitchMessage ParseMessage(FirehoseMessage message)
        {
            var tags = TwitchMessageParser.ParseTags(message.Tags);
            var channel = message.Room.Trim('#', '@');

            var messageType = TwitchMessageParser.GetType(message.Command);
            var noticeType = GetNoticeType(messageType, tags, message.Body);
            var twitchMessage = new TwitchMessage(messageType, noticeType, tags, message.Body, message.Nick, channel);

            ProcessTwitchMessage(twitchMessage, tags, message);
            return twitchMessage;
        }

        private static TwitchChatNoticeType GetNoticeType(IrcMessageType messageType, Dictionary<TwitchChatTag, object> tags, string messageBody)
        {
            switch (messageType)
            {
                case IrcMessageType.Roomstate:
                    object value;
                    if (tags.TryGetValue(TwitchChatTag.SubsOnly, out value))
                    {
                        return (bool) value ? TwitchChatNoticeType.SubsOnlyOn : TwitchChatNoticeType.SubsOnlyOff;
                    }
                    if (tags.TryGetValue(TwitchChatTag.EmoteOnly, out value))
                    {
                        return (bool) value ? TwitchChatNoticeType.EmoteOnlyOn : TwitchChatNoticeType.EmoteOnlyOff;
                    }
                    if (tags.TryGetValue(TwitchChatTag.R9k, out value))
                    {
                        return (bool) value ? TwitchChatNoticeType.R9KOn : TwitchChatNoticeType.R9KOff;
                    }
                    if (tags.TryGetValue(TwitchChatTag.Slow, out value))
                    {
                        return ((int) value) == 0 ? TwitchChatNoticeType.SlowModeOff : TwitchChatNoticeType.SlowModeOn;
                    }
                    break;
                case IrcMessageType.UserNotice:
                    var msgID = (string) tags.GetValueOrDefault(TwitchChatTag.MessageId);
                    if (msgID == "sub") {
                        return TwitchChatNoticeType.Sub;
                    }
                    return TwitchChatNoticeType.Resub;
                case IrcMessageType.ClearChat:
                    return string.IsNullOrEmpty(messageBody) ? TwitchChatNoticeType.ChatCleared : TwitchChatNoticeType.UserTimedOut;
                case IrcMessageType.HostTarget:
                    return messageBody.StartsWith("-") ? TwitchChatNoticeType.HostOff : TwitchChatNoticeType.HostOn;
            }

            return TwitchChatNoticeType.Unknown;
        }

        private static readonly Dictionary<TwitchChatNoticeType, string> _textMap = new Dictionary<TwitchChatNoticeType, string>
        {
            {TwitchChatNoticeType.SubsOnlyOn, "This room is now in subscribers-only mode."},
            {TwitchChatNoticeType.SubsOnlyOff, "This room is no longer in subscribers-only mode."},

            {TwitchChatNoticeType.EmoteOnlyOn, "This room is now in emote-only mode."},
            {TwitchChatNoticeType.EmoteOnlyOff, "This room is no longer in emote-only mode."},

            {TwitchChatNoticeType.R9KOn, "This room is now in r9k mode."},
            {TwitchChatNoticeType.R9KOff, "This room is no longer in r9k mode."},

            {TwitchChatNoticeType.SlowModeOn, "This room is now in slow mode. You may send messages every {0} seconds."},
            {TwitchChatNoticeType.SlowModeOff, "This room is no longer in slow mode."},

            {TwitchChatNoticeType.ChatCleared, "Chat was cleared by a moderator"},
            {TwitchChatNoticeType.UserTimedOut, "{0}"},
            {TwitchChatNoticeType.Resub, "{0}"},

            {TwitchChatNoticeType.UserSubscribed, "{0}"},
            {TwitchChatNoticeType.HostOn, "Now hosting {0}."},
            {TwitchChatNoticeType.HostOff, "Exited host mode."},
            {TwitchChatNoticeType.HostTargetWentOffline, "{0} has gone offline. Exiting host mode."},

            {TwitchChatNoticeType.FollowersOnlyOn, "This room is now in {0} followers-only mode." },
            {TwitchChatNoticeType.FollowersOnlyOnZero, "This room is now in followers-only mode." },
            {TwitchChatNoticeType.FollowersOnlyOff, "This room is no longer in followers-only mode" },

        };

        private static void ProcessTwitchMessage(TwitchMessage twitchMessage, Dictionary<TwitchChatTag, object> tags, FirehoseMessage message)
        {
            if (twitchMessage.NoticeType != TwitchChatNoticeType.Unknown)
            {
                twitchMessage.Data = GetNoticeMessage(twitchMessage.NoticeType, message, tags);

                if (twitchMessage.NoticeType == TwitchChatNoticeType.UserTimedOut)
                {
                    var targetUserID = (long) (tags.GetValueOrDefault(TwitchChatTag.TargetUserID) ?? 0L);
                    if (targetUserID <= 0)
                    {
                        return;
                    }
                    twitchMessage.UserID = targetUserID;
                    twitchMessage.Username = message.Body;
                }
            }
        }

        private static string GetNoticeMessage(TwitchChatNoticeType noticeType, FirehoseMessage data, Dictionary<TwitchChatTag,object> tags)
        {
            string formatText;
            if (!_textMap.TryGetValue(noticeType, out formatText))
            {
                return data.Body;
            }

            var formatArgs = new List<object>();

            switch (noticeType)
            {
                case TwitchChatNoticeType.Resub:
                    formatArgs.Add(tags.GetValueOrDefault(TwitchChatTag.SystemMessage) ?? string.Empty);
                    break;
                case TwitchChatNoticeType.SlowModeOn:
                    formatArgs.Add(tags.GetValueOrDefault(TwitchChatTag.Slow) ?? 120);
                    break;
                case TwitchChatNoticeType.UserTimedOut:
                    formatArgs.Add(tags.GetValueOrDefault(TwitchChatTag.BanReason));
                    break;
                case TwitchChatNoticeType.HostOn:
                    formatArgs.Add(data.Body.Split()[0]);
                    break;
                case TwitchChatNoticeType.HostTargetWentOffline:
                    formatArgs.Add("TEST");
                    break;
                case TwitchChatNoticeType.FollowersOnlyOn:
                    formatArgs.Add(data.Body);
                    break;
                default:
                    return formatText;
            }

            return string.Format(formatText, formatArgs.ToArray());
        }
    }
}
