﻿using System;
using Curse.Logging;

namespace Curse.Friends.TwitchService.Chat.Irc
{
    public abstract class BaseTwitchConnection
    {
        protected TwitchClient _client;
        protected readonly LogCategory Logger;
        private volatile bool _isShuttingDown;

        protected BaseTwitchConnection(TwitchLoginInfo loginInfo, LogCategory logCategory)
        {
            Logger = logCategory;
            _client = new TwitchClient(loginInfo);

            _client.RawMessageReceived += IrcMessageReceived;
            _client.Error += ErrorCallback;
            _client.Disconnected += DisconnectedCallback;
        }

        public bool IsConnected { get; private set; }

        public bool Connect()
        {
            var connected = _client.Connect();
            if (connected)
            {
                Logger.Debug("Connected to IRC server.");
                CustomConnectedCallback();
            }
            IsConnected = connected;
            return connected;
        }

        public void Close()
        {
            _isShuttingDown = true;
            
            // Clean up
            _client.Disconnected -= DisconnectedCallback;
            _client.Error -= ErrorCallback;
            _client.RawMessageReceived -= IrcMessageReceived;

            if (IsConnected)
            {
                IsConnected = false;
                DisconnectedCallback(this, EventArgs.Empty);
            }

            if (_client != null && _client.IsConnected)
            {
                _client.Stop();
            }

            if (_client != null)
            {
                _client.Dispose();
            }
        }

        protected virtual void CustomConnectedCallback()
        {
            
        }

        private void IrcMessageReceived(object sender, RawMessageReceivedEventArgs e)
        {
            CustomIrcMessageCallback(e);

            var message = IrcMessageParser.ParseMessage(e.Message);
            if (message != null)
            {
                CustomTwitchMessageCallback(message);
            }
        }

        protected virtual void CustomIrcMessageCallback(RawMessageReceivedEventArgs e)
        {
#if DEBUG
            Console.WriteLine("RAW: " + e.Message);
#endif
        }

        protected virtual void CustomTwitchMessageCallback(TwitchMessage message)
        {
        }

        private void ErrorCallback(object sender, IrcErrorEventArgs e)
        {
            if (e.Error == null || e.Error.Message.Equals("Sequence contains no matching element") || e.Error.Message.Equals("Sequence contains more than one matching element"))
            {
                return;
            }

            Logger.Warn(e.Error, "General error in TwitchBot");
        }

        private void DisconnectedCallback(object sender, EventArgs e)
        {
            IsConnected = false;
            CustomDisconnectedCallback(_isShuttingDown);
            Logger.Debug("IRC client disconnected");
        }

        protected virtual void CustomDisconnectedCallback(bool isShuttingDown)
        {
            
        }
    }
}
