﻿using System.Collections.Generic;
using System.Text.RegularExpressions;
using Curse.Extensions;
using Curse.Friends.Enums;
using Curse.Friends.TwitchService.Chat.Parsing;
using Curse.Logging;

namespace Curse.Friends.TwitchService.Chat.Irc
{
    public static class IrcMessageParser
    {
        private static readonly Regex _chatMessageRegex =
            new Regex(@"^(?:@(?<tags>(?:(?:[\w-]*=[^ ;]*?);?)*) )?(?::(?<userinfo>[\w-]*!?[a-zA-Z0-9@_\.-]+) |jtv )?(?<command>[a-zA-Z0-9]+) (?:(?<username>[a-zA-Z0-9]+) )?(?:= )?(?:[#@]?(?<channel>[a-zA-Z0-9_-]+|\*)? ?)?(?::?(?<data>.*))?$", RegexOptions.Compiled);

        public static TwitchMessage ParseMessage(string message)
        {
            var match = _chatMessageRegex.Match(message);
            if (match.Success)
            {
                var tagGroup = match.Groups["tags"];
                var dataGroup = match.Groups["data"];
                var commandGroup = match.Groups["command"];
                var channelGroup = match.Groups["channel"];

                var tags = tagGroup == null ? new Dictionary<TwitchChatTag, object>() : TwitchMessageParser.ParseTags(tagGroup.Value);
                var data = dataGroup == null ? null : match.Groups["data"].Value;
                var channel = channelGroup == null ? null : channelGroup.Value.Trim('#', '@');
                var messageType = commandGroup == null ? IrcMessageType.Unknown : TwitchMessageParser.GetType(commandGroup.Value);
                var noticeType = TwitchMessageParser.GetNoticeType((string) tags.GetValueOrDefault(TwitchChatTag.MessageId), channel, data);
                var username = GetUsername(match.Groups["userinfo"].Value);
                var twitchMessage = new TwitchMessage(messageType, noticeType, tags, data, username, channel);
                return twitchMessage;
            }
            else
            {
                Logger.Warn("Failed to parse Twitch Message", message);
            }
            return null;
        }


        private static string GetUsername(string userInfoString)
        {
            var index = userInfoString.IndexOf('!');
            if (index < 0)
            {
                return null;
            }
            return userInfoString.Substring(0, index);
        }

        public static bool IsRoomBasedNotice(TwitchChatNoticeType type)
        {
            switch (type)
            {
                case TwitchChatNoticeType.SlowModeOn:
                case TwitchChatNoticeType.SlowModeOff:
                case TwitchChatNoticeType.EmoteOnlyOn:
                case TwitchChatNoticeType.EmoteOnlyOff:
                case TwitchChatNoticeType.R9KOn:
                case TwitchChatNoticeType.R9KOff:
                case TwitchChatNoticeType.SubsOnlyOn:
                case TwitchChatNoticeType.SubsOnlyOff:
                case TwitchChatNoticeType.UserSubscribed:
                case TwitchChatNoticeType.HostOn:
                case TwitchChatNoticeType.HostOff:
                case TwitchChatNoticeType.HostTargetWentOffline:
                    return true;
                default:
                    return false;
            }
        }
    }
}
